/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "text.h"
#include "font.h"

#include "util/loading/resource_manager.h"
#include "debug.h"

using namespace std;

/**
 * @brief creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
Text::Text(const std::string& fontFile, unsigned int textSize)
{
  this->setClassID(CL_TEXT, "Text");

  // initialize this Text
  this->_font = NULL;
  this->_size = textSize;
  this->setSizeY2D(textSize);
  this->_color = TEXT_DEFAULT_COLOR;

  this->setAlignment(TEXT_DEFAULT_ALIGNMENT);

  this->setFont(fontFile, FONT_DEFAULT_RENDER_SIZE);
}

Text::Text(const Text& text)
{
  this->setClassID(CL_TEXT, "Text");
  this->_font = NULL;

  *this = text;
}


/**
 * @brief deletes a Text out of memory
 */
Text::~Text()
{
  if (this->_font != NULL && this->_font != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(this->_font);
}

/**
 * @brief compare the Text with another Text.
 * @param text the Text to compare.
 * @returns true if all the properties Match.
 */
bool Text::operator==(const Text& text) const
{
  return (this->_text == text._text &&
          this->_size == text._size &&
          this->_font == text._font &&
          this->_color == text._color);
}

/**
 * @brief compare this Text's internal String with the text.
 * @param text the Comparator Text.
 * @returns true on a match.
 */
bool Text::operator==(const std::string& text) const
{
  return (this->_text == text);
}

/**
 * @brief Copies the properties of one text onto the other one.
 * @param text: the Text to apply to this one.
 * @returns This-reference.
 */
Text& Text::operator=(const Text& text)
{
  this->_size = text._size;
  this->_color = text._color;
  this->setAlignment(text.getAlignment());
  if (this->_font != NULL)
    ResourceManager::getInstance()->unload(this->_font);

  this->_font = (Font*)ResourceManager::getInstance()->copy( text._font ); //!< HACK

  this->_text = text._text;
  return *this;
}

/**
 * @brief Sets a new Text to the font
 * @param text the new text to set
 */
void Text::setText(const std::string& text)
{
  this->_text = text;
  this->setupTextWidth();
}

/**
 * @brief append some text to the already existing Text.
 * @param appendText The text to append to this Text.
 */
void Text::append(const std::string& appendText)
{
  this->_text += appendText;
  this->setupTextWidth();
}

/**
 * @brief appends one Character to the String.
 */
void Text::appendCharacter(char character)
{
  this->_text += character;
  this->setupTextWidth();
}


/**
 * @brief append some text to the already existing Text.
 * @param appendText The text to append to this Text.
 */
const std::string& Text::operator <<(const std::string& appendText)
{
  this->append(appendText);
  return this->_text;
}

/**
 * @brief removes char characters from the Text.
 *
 * @note this function checks, if the count can be removed, and if so does it.
 * Otherwise the maximum count of characters will be removed.
 */
void Text::removeCharacters(unsigned int chars)
{
  if (this->_text.size() > chars)
    this->_text.resize(this->_text.size()-chars);
  else if (!this->_text.empty())
    this->_text.clear();
  this->setupTextWidth();
}


/**
 * @brief clears the Text Line (empies it).
 */
void Text::clear()
{
  this->_text.clear();
  this->setupTextWidth();
}

/**
 * @brief sets the Font of this Text to font from fontFile
 * @param fontFile the File to load the Font from.
 * @param fontSize the Size of the Font
 */
void Text::setFont(const std::string& fontFile, unsigned int fontSize)
{
  Font* newFont = NULL;
  Font* oldFont = this->_font;

  // load a new Font
  if (!fontFile.empty())
  {
    newFont = (Font*)ResourceManager::getInstance()->load(fontFile, TTF, RP_GAME, (int)fontSize);
    if (newFont == NULL)
    {
      newFont = Font::getDefaultFont();
      PRINTF(2)("Font %s could not be loaded, probably file not found\n", fontFile.c_str());
    }
  }
  if (newFont == NULL)
    newFont = Font::getDefaultFont();
  assert(newFont != NULL);

  // unloading the Font if we alrady have one loaded.
  this->_font = newFont;
  if (oldFont != NULL && oldFont != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(oldFont);

  this->setupTextWidth();
}

/**
 * @brief sets the Size of the Font
 * @param size :the size of the Text
 */
void Text::setSize(float size)
{
  this->_size = size;
  this->setSizeY2D(size);
  this->setupTextWidth();
}


/**
 * @brief draws the Text
 */
void Text::draw() const
{
  if (unlikely(this->_text.empty()))
    return;
  glPushMatrix();
  glPushAttrib(GL_ENABLE_BIT);
  // transform for alignment.
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->getSizeX2D(), 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->getSizeX2D()/2, 0, 0);

  // drawing this Text.
  // setting the Blending effects
  glColor4fv(&this->_color[0]);


  glActiveTexture(GL_TEXTURE0);

  glEnable(GL_BLEND);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, GL_MODULATE );

  glBindTexture(GL_TEXTURE_2D, this->_font->getTexture());
  glTranslatef(getAbsCoor2D().x, getAbsCoor2D().y, 0);
  glRotatef(this->getAbsDir2D(), 0, 0, 1);

  Glyph* tmpGlyph;
  float posX = 0.0f;
  glBegin(GL_QUADS);
  for (unsigned int i = 0; i < this->_text.size(); i++)
  {
    if(likely((tmpGlyph = this->font()->getGlyphArray()[this->_text[i]]) != NULL))
    {
      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->maxX*this->size(), 0);

      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->maxX*this->size(), this->size());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->minX*this->size(), this->size());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->minX*this->size(), 0);

      posX += tmpGlyph->advance * this->size();
    }
  }
  glEnd();
  glPopAttrib();
  glPopMatrix();
}


/**
 * @brief setting up the Text-Width.
 */
void Text::setupTextWidth()
{
  float width = 0;
  for (unsigned int i = 0; i < this->_text.size(); i++)
    if(this->_font->getGlyphArray()[this->_text[i]] != NULL)
      width += this->_font->getGlyphArray()[this->_text[i]]->advance;
  this->setSizeX2D(width * this->size());
}


/**
 * @brief prints out some nice debug information about this text
 */
void Text::debug() const
{
  PRINT(0)("=== TEXT: %s (with Font:'%s')  displaying %s ===\n", this->getName(), this->_font->getName(), this->_text.c_str());
  PRINT(0)("Color: r=%0.2f g=%0.2f b=%0.2f a=%0.2f\n", this->_color.r(), this->_color.g(), this->_color.b(), this->_color.a());
}

