/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GUI

#include "glgui_inputline.h"

namespace OrxGui
{
  /**
   * @brief standard constructor
   */
  GLGuiInputLine::GLGuiInputLine ()
  {
    this->init();
  }


  /**
   * @brief standard deconstructor
   */
  GLGuiInputLine::~GLGuiInputLine()
  {}

  float GLGuiInputLine::repeatDelay = 0.3f;
  float GLGuiInputLine::repeatRate  = 0.01f;


  /**
   * @brief initializes the GUI-element
   */
  void GLGuiInputLine::init()
  {
    this->setClassID(CL_GLGUI_INPUTLINE, "GLGuiInputLine");

    this->setFocusable(true);

    this->_clearOnEnter = false;
    this->_text.setParent2D(this);
    this->_text.setRelCoor2D(4,4);
    this->_text.setFont("fonts/final_frontier.ttf", 20);
    this->_text.setColor(this->frontColor());
    this->_text.setVisibility(false);
    this->resize();

  }


  /**
   * @brief sets the Text of the InputLine
   * @param text The new Text.
   */
  void GLGuiInputLine::setText(const std::string& text)
  {
    this->_text.setText(text);
    this->changedText();
  }

  /**
   * @brief appends text to the InputLine
   * @param appendText the Text to append
   */
  void GLGuiInputLine::append(const std::string& appendText)
  {
    this->_text.append(appendText);
    this->changedText();
  }


  /**
   * @brief appends a Character to the InputLine
   * @param character the Character to append.
   */
  void GLGuiInputLine::appendCharacter(char character)
  {
    this->_text.appendCharacter(character);
    this->changedText();
  }


  /**
   * @brief Removes Characters from the InputLine
   * @param chars The count of characters to remove
   */
  void GLGuiInputLine::removeCharacters(unsigned int chars)
  {
    this->_text.removeCharacters(chars);
    this->changedText();
  }

  void GLGuiInputLine::clear()
  {
    this->_text.clear();
    this->changedText();
  }

  /**
   * @brief If the Text has been changed this function is called.
   *
   * This Function also emits the Signal textChanged.
   */
  void GLGuiInputLine::changedText()
  {
    this->resize();
    this->setFrontColor(Color(1,1,1,1), true);
    this->setFrontColor(Color(0,1,0,1));
    emit(this->textChanged(this->_text.getText()));
  }


  /**
   * removes the focus from this Widget.
   */
  void GLGuiInputLine::removedFocus()
  {
    GLGuiWidget::removedFocus();
    this->pressedKey = 0;
    this->pressedKeyName = 0;
  }

  /**
   * @brief handles the EnterPush-Event.
   *
   * Emmits th EnterPushed Signal with the current Line,
   * and if _clearOnEnter is pushed clears the Line.
   */
  void GLGuiInputLine::pushEnter()
  {
    emit(this->enterPushed(this->_text.getText()));
    if (this->_clearOnEnter)
      this->clear();
  }


  /**
   * @brief Processes an Event.
   * @param event The event to be processed
   * @return true if the event was catched.
   */
  bool GLGuiInputLine::processEvent(const Event& event)
  {
    if (event.bPressed)
    {
      if (event.type == SDLK_BACKSPACE)
      {
        this->delayNext = GLGuiInputLine::repeatDelay;
        this->pressedKey = SDLK_BACKSPACE;
        this->pressedKeyName = SDLK_BACKSPACE;
        this->removeCharacters(1);
        return true;
      }
      else if(event.type == SDLK_RETURN || event.type == SDLK_KP_ENTER)
      {
        this->pushEnter();
        return true;
      }
      // any other keyboard key
      else if (likely(event.type < 127))
      {
        this->delayNext = GLGuiInputLine::repeatDelay;

        this->appendCharacter(event.x);
        this->pressedKey = event.type;
        this->pressedKeyName = event.x;
        return true;
      }
    }
    else // if(!event.bPressed)
    {
      if (this->pressedKey == event.type)
      {
        this->pressedKeyName = 0;
        this->pressedKey = 0;
        this->delayNext = 0.0;
        return true;
      }
    }
    return false;
  }


  /**
   * @brief Resizes the Widget to the new Size-constraints.
   */
  void GLGuiInputLine::resize()
  {
    this->_text.setRelCoor2D(this->borderLeft(), this->borderTop());
    this->setSize2D( this->_text.getSize2D() + Vector2D(borderLeft() + borderRight(), borderTop() + borderBottom()));
    GLGuiWidget::resize();
/*    this->frontRect().setTopLeft(borderLeft(), borderTop());
    this->frontRect().setSize(this->getSize2D() - Vector2D(borderLeft() + borderRight(), borderTop() + borderBottom()));*/
  }

  void GLGuiInputLine::updateFrontColor()
  {
    this->_text.setColor(this->frontColor());
  }

  void GLGuiInputLine::hiding()
  {
    this->_text.setVisibility(false);
  }

  void GLGuiInputLine::showing()
  {
    this->_text.setVisibility(true);
  }

  /**
   * @brief ticks the InputLine
   * @param dt the time passed.
   */
  void GLGuiInputLine::tick(float dt)
  {
    GLGuiWidget::tick(dt);
    if (this->delayNext > 0.0)
      this->delayNext -= dt;
        else if (this->pressedKey != SDLK_FIRST )
    {
      this->delayNext = GLGuiInputLine::repeatRate;
      switch (this->pressedKey)
      {
        case SDLK_BACKSPACE:
          this->removeCharacters(1);
          break;
        default:
        {
          if (likely(this->pressedKey < 127))
            this->appendCharacter(this->pressedKeyName);
        }
      }
    }


  }

  /**
   * @brief draws the GLGuiInputLine
   */
  void GLGuiInputLine::draw() const
  {
    this->beginDraw();
    GLGuiWidget::draw();

//     this->frontMaterial().select();
//     GLGuiWidget::drawRect(this->frontRect());

    this->endDraw();
  }
}
