/*
  orxonox - the future of 3D-vertical-scrollers

  Copyright (C) 2004 orx

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

### File Specific:
  main-programmer: hdavid, amaechler
*/

#include "fog_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "shell_command.h"

SHELL_COMMAND(activate, FogEffect, activateFog);
SHELL_COMMAND(deactivate, FogEffect, deactivateFog);
SHELL_COMMAND(fadein, FogEffect, fadeInFog);
SHELL_COMMAND(fadeout, FogEffect, fadeOutFog);

// TODO: Fix fades

using namespace std;

CREATE_FACTORY(FogEffect, CL_FOG_EFFECT);

FogEffect::FogEffect(const TiXmlElement* root) {
    this->setClassID(CL_FOG_EFFECT, "FogEffect");

    this->init();

    if (root != NULL)
        this->loadParams(root);

    if (this->fogActivate)
        this->activate();
}


FogEffect::~FogEffect() {
    this->deactivate();
}


void FogEffect::loadParams(const TiXmlElement* root) {
    WeatherEffect::loadParams(root);

    LoadParam(root, "mode", this, FogEffect, setFogMode);
    LoadParam(root, "density", this, FogEffect, setFogDensity);
    LoadParam(root, "range", this, FogEffect, setFogRange);
    LoadParam(root, "color", this, FogEffect, setFogColor);
    LoadParam(root, "fadeinduration", this, FogEffect, setFogFadeIn);
    LoadParam(root, "fadeoutduration", this, FogEffect, setFogFadeOut);

    LOAD_PARAM_START_CYCLE(root, element);
    {
        LoadParam_CYCLE(element, "option", this, FogEffect, setFogOption);
    }
    LOAD_PARAM_END_CYCLE(element);
}

void FogEffect::init() {
    // default values
    this->fogMode = GL_LINEAR;
    this->fogDensity = 0.03;
    this->fogStart = 0;
    this->fogEnd = 50;
    this->colorVector = Vector(0.3, 0.3, 0.3);

    // init variables
    this->fogFadeInDuration = 0;
    this->fogFadeOutDuration = 0;
    this->fogFadeDensity = 0;
    this->localTimer = 0;
    this->fogActivate = false;
    this->fogFadeInActivate = false;
    this->fogFadeOutActivate = false;

}


void FogEffect::activate() {
    PRINTF(0)( "Enabling FogEffect, mode: %i, density: %f, start: %f, end: %f, color %f, %f, %f\n", this->fogMode, this->fogDensity, this->fogStart, this->fogEnd, this->colorVector.x, this->colorVector.y, this->colorVector.z);

    this->fogActivate = true;

    GLfloat fogColor[4] = { colorVector.x, colorVector.y, colorVector.z, 1.0};
    glFogi(GL_FOG_MODE, this->fogMode);
    glFogfv(GL_FOG_COLOR, fogColor);
    glHint(GL_FOG_HINT, GL_DONT_CARE);
    glFogf(GL_FOG_DENSITY, this->fogDensity);
    glFogf(GL_FOG_START, this->fogStart);
    glFogf(GL_FOG_END, this->fogEnd);

    glEnable(GL_FOG);

}


void FogEffect::deactivate() {
    PRINTF(0)("Deactivating FogEffect\n");

    this->fogFadeInActivate = false;
    this->fogFadeOutActivate = false;
    this->fogActivate = false;

    glDisable(GL_FOG);

}

void FogEffect::draw() const {

    if (!this->fogActivate)
        return;

    // If Fog Fade
    if ( this->fogFadeInActivate || this->fogFadeOutActivate )
        glFogf(GL_FOG_DENSITY, this->fogFadeDensity);

}

void FogEffect::tick(float dt) {
    if (!this->fogActivate)
        return;

    if ( this->fogFadeInActivate ) {
        this->localTimer += dt;
        this->fogFadeDensity = ( this->localTimer / this->fogFadeInDuration ) * this->fogDensity;

        if ( this->localTimer >= this->fogFadeOutDuration )
            this->fogFadeInActivate = false;
    }

    if ( this->fogFadeOutActivate ) {
        this->localTimer += dt;
        this->fogFadeDensity = 1 - (( this->localTimer / this->fogFadeInDuration ) * this->fogDensity);

        if ( this->localTimer >= this->fogFadeOutDuration )
            this->deactivate();
    }
}

void FogEffect::fadeInFog() {

    // If Fog is already fading out, stop it
    this->fogFadeOutActivate = false;

    // If Fog is already on, turn it off first
    if (this->fogActivate)
        this->deactivate();

    // If no manual FadeIn value was set, set a default value
    if (!this->fogFadeInDuration > 0)
        this->fogFadeInDuration = 20;

    // Reset local timer
    this->localTimer = 0;

    // set FogFadeIn activate
    this->fogFadeInActivate = true;

    // Activate Fog
    this->activate();

}


void FogEffect::fadeOutFog() {

    // If Fog is already fading in, stop it
    this->fogFadeInActivate = false;

    // If Fog is off, turn it on first
    if (!this->fogActivate)
        this->activate();

    // If no manual FadeOut value was set, set a default value
    if (!this->fogFadeOutDuration > 0)
        this->fogFadeOutDuration = 20;

    // set FogFadeOut activate
    this->fogFadeOutActivate = true;

    // Reset local timer
    this->localTimer = 0;

}


GLint FogEffect::stringToFogMode(const std::string& mode) {
    if(mode == "GL_LINEAR")
        return GL_LINEAR;
    else if(mode == "GL_EXP")
        return GL_EXP;
    else if(mode == "GL_EXP2" )
        return GL_EXP2;
    else
        return -1;
}


