/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: hdavid, amaechler
*/

#include "lightening_effect.h"

#include "state.h"
#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "effects/billboard.h"

#include "glincl.h"
#include "parser/tinyxml/tinyxml.h"

#include "shell_command.h"
#include "light.h"

SHELL_COMMAND(activate, LighteningEffect, activateLightening);
SHELL_COMMAND(deactivate, LighteningEffect, deactivateLightening);

using namespace std;

CREATE_FACTORY(LighteningEffect, CL_LIGHTENING_EFFECT);

// TODO: FIx Lightening with Fog enabled

LighteningEffect::LighteningEffect(const TiXmlElement* root)
{
	this->setClassID(CL_LIGHTENING_EFFECT, "LighteningEffect");
  
	this->init();

	if (root != NULL)
		this->loadParams(root);

	if(this->lighteningActivate)
		this->activate();
}

LighteningEffect::~LighteningEffect()
{
	this->deactivate();
}

void LighteningEffect::loadParams(const TiXmlElement* root)
{
	WeatherEffect::loadParams(root);

	LoadParam(root, "coord", this, LighteningEffect, coord);
	LoadParam(root, "frequency", this, LighteningEffect, setFlashFrequency);
	LoadParam(root, "const-time", this, LighteningEffect, setFlashConstTime);
	LoadParam(root, "rising-time", this, LighteningEffect, setFlashRisingTime);
	LoadParam(root, "size", this, LighteningEffect, setFlashSize);
	LoadParam(root, "seed", this, LighteningEffect, setFlashSeed);

	LOAD_PARAM_START_CYCLE(root, element);
	{
		LoadParam_CYCLE(element, "option", this, LighteningEffect, setLighteningOption);
	}
	LOAD_PARAM_END_CYCLE(element);
}


bool LighteningEffect::init()
{
	//default values
	this->lighteningActivate = false;
	
	this->time = 0.0;
	this->flashFrequency = 4.0f;
	this->mainFrequency = 4.0f;
	this->flashConstTime = 0.1f;
	this->flashRisingTime = 0.03f;
	
	this->width = 700.0f;
	this->height = 250.0f;
	this->seedWidth = 50.0f;
	this->seedHeight = 50.0f;
	this->bNewCoordinate = false;
  this->lighteningMove = false;
	
	this->seedX = 500.f;
	this->seedZ = 1000.0f;
	this->seedTime = 2.0f;
  
  this->mainPosX = 3000;
  this->mainPosY = 900;
  this->mainPosZ = 0;
	
	// initialize lightening textures
	this->billboard[0] = new Billboard(NULL);
	this->billboard[0]->setTexture("maps/lightning_bolt1.png");
	this->billboard[0]->setSize(this->width, this->height);
	this->billboard[0]->setVisibiliy(false);
	
	this->billboard[1] = new Billboard(NULL);
	this->billboard[1]->setTexture("maps/lightning_bolt2.png");
	this->billboard[1]->setSize(this->width, this->height);
	this->billboard[1]->setVisibiliy(false);
	
	this->billboard[2] = new Billboard(NULL);
	this->billboard[2]->setTexture("maps/lightning_bolt3.png");
	this->billboard[2]->setSize(this->width, this->height);
	this->billboard[2]->setVisibiliy(false);
	
	this->billboard[3] = new Billboard(NULL);
	this->billboard[3]->setTexture("maps/lightning_bolt4.png");
	this->billboard[3]->setSize(this->width, this->height);
	this->billboard[3]->setVisibiliy(false);
	
	if (this->lighteningMove) {
		this->cameraCoor = State::getCameraNode()->getAbsCoor();
		this->billboard[0]->setAbsCoor(this->cameraCoor.x+3000,900,this->cameraCoor.z+0);
		this->billboard[1]->setAbsCoor(this->cameraCoor.x+3000,900,this->cameraCoor.z+0);
		this->billboard[2]->setAbsCoor(this->cameraCoor.x+3000,900,this->cameraCoor.z+0);
		this->billboard[3]->setAbsCoor(this->cameraCoor.x+3000,900,this->cameraCoor.z+0);
	} else {
		this->billboard[0]->setAbsCoor(3000,900,0);
		this->billboard[1]->setAbsCoor(3000,900,0);
		this->billboard[2]->setAbsCoor(3000,900,0);
		this->billboard[3]->setAbsCoor(3000,900,0);
	}
	
	this->flashLight = new Light();
	this->flashLight->setDiffuseColor(0,0,0);
	this->flashLight->setSpecularColor(0,0,0);
	
	/*
	this->soundSource = NULL;
	this->thunderBuffer = NULL;
	
	this->soundSource.setSourceNode(this);
	
	//load sound
	if (this->thunderBuffer != NULL)
	ResourceManager::getInstance()->unload(this->thunderBuffer);
	this->thunderBuffer = (OrxSound::SoundBuffer*)ResourceManager::getInstance()->load("sound/thunder.wav", WAV);*/

	return 0;
}

void LighteningEffect::coord(float x, float y, float z)
{
	if (this->lighteningMove) {
		this->cameraCoor = State::getCameraNode()->getAbsCoor();
		this->billboard[0]->setAbsCoor(this->cameraCoor.x+x, y, this->cameraCoor.z+z);
		this->billboard[1]->setAbsCoor(this->cameraCoor.x+x, y, this->cameraCoor.z+z);
		this->billboard[2]->setAbsCoor(this->cameraCoor.x+x, y, this->cameraCoor.z+z);
		this->billboard[3]->setAbsCoor(this->cameraCoor.x+x, y, this->cameraCoor.z+z);
	} else {
		this->billboard[0]->setAbsCoor(x, y, z);
		this->billboard[1]->setAbsCoor(x, y, z);
		this->billboard[2]->setAbsCoor(x, y, z);
		this->billboard[3]->setAbsCoor(x, y, z);
	}

	this->mainPosX = x;
	this->mainPosY = y;
	this->mainPosZ = z;
}


void LighteningEffect::setFlashSize(float width, float height, float seedWidth, float seedHeight)
{
	this->width = width;
	this->height = height;
	this->seedWidth = seedWidth;
	this->seedHeight = seedHeight;

	this->billboard[0]->setSize(this->width, this->height);
	this->billboard[1]->setSize(this->width, this->height);
	this->billboard[2]->setSize(this->width, this->height);
	this->billboard[3]->setSize(this->width, this->height);
}


bool LighteningEffect::activate()
{
	PRINTF(0)( "Activating LighteningEffect\n" );
	this->lighteningActivate = true;

	this->time = 0;

	return 0;
}


bool LighteningEffect::deactivate()
{
	PRINTF(0)("Deactivating LighteningEffect\n");
	this->lighteningActivate = false;

	this->billboard[0]->setVisibiliy(false);
	this->billboard[1]->setVisibiliy(false);
	this->billboard[2]->setVisibiliy(false);
	this->billboard[3]->setVisibiliy(false);

	return 0;
}

void LighteningEffect::tick (float dt)
{
	if(!lighteningActivate)
		return;
	
	this->time += dt;

	// TODO: Make random flashing with short light dingsda:)
	
	if( this->time > this->flashFrequency) {
	
		this->billboard[0]->setVisibiliy(true);
		this->time = 0.0f;
	
		this->flashLight->setAbsCoor(this->billboard[0]->getAbsCoor().x, this->billboard[0]->getAbsCoor().y, this->billboard[0]->getAbsCoor().z);

		// flash environmental lightening
		this->flashLight->setDiffuseColor(1,1,1);
		this->flashLight->setSpecularColor(1,1,1);
	
		//this->soundSource.play(this->thunderBuffer);

	} else if( this->billboard[3]->isVisible() && this->time > this->flashConstTime) {
	
		this->billboard[3]->setVisibiliy(false);
		this->time = 0.0f;
		this->flashLight->setDiffuseColor(0,0,0);
		this->flashLight->setSpecularColor(0,0,0);
		this->bNewCoordinate = true;

	}
	
	if( this->billboard[2]->isVisible() && this->time > this->flashRisingTime) {
	
		this->billboard[2]->setVisibiliy(false);
		this->billboard[3]->setVisibiliy(true);
		// this->flashLight->setDiffuseColor(1,1,1);
		// this->flashLight->setSpecularColor(1,1,1);
		
	} else if( this->billboard[1]->isVisible() && this->time > this->flashRisingTime*2/3 ) {
	
		this->billboard[1]->setVisibiliy(false);
		this->billboard[2]->setVisibiliy(true);
		//this->flashLight->setDiffuseColor(0,0,0);
		//this->flashLight->setSpecularColor(0,0,0);
		
	} else if( this->billboard[0]->isVisible() && this->time > this->flashRisingTime*1/3 ) {
	
		this->billboard[0]->setVisibiliy(false);
		this->billboard[1]->setVisibiliy(true);
		//this->flashLight->setDiffuseColor(1,1,1);
		//this->flashLight->setSpecularColor(1,1,1);
		
	}
	
	if( this->bNewCoordinate) {
		float posX, posZ;

		if(this->lighteningMove) {
		
			this->cameraCoor = State::getCameraNode()->getAbsCoor();
			posX = this->mainPosX - this->seedX * (float)rand()/(float)RAND_MAX + this->cameraCoor.x;
			posZ = this->mainPosZ + this->seedZ * (float)rand()/(float)RAND_MAX + this->cameraCoor.z;

		} else {
		
			posX = this->mainPosX - this->seedX * (float)rand()/(float)RAND_MAX;
			posZ = this->mainPosZ + this->seedZ * (float)rand()/(float)RAND_MAX;

		}
	
		this->billboard[0]->setAbsCoor(posX, this->mainPosY, posZ);
		this->billboard[1]->setAbsCoor(posX, this->mainPosY, posZ);
		this->billboard[2]->setAbsCoor(posX, this->mainPosY, posZ);
		this->billboard[3]->setAbsCoor(posX, this->mainPosY, posZ);
		
		this->flashFrequency = this->mainFrequency + this->seedTime * (float)rand()/(float)RAND_MAX;
		
		float w = this->width + this->seedWidth * (float)rand()/(float)RAND_MAX;
		float h = this->height + this->seedHeight * (float)rand()/(float)RAND_MAX;
		
		this->billboard[0]->setSize(w, h);
		this->billboard[1]->setSize(w, h);
		this->billboard[2]->setSize(w, h);
		this->billboard[3]->setSize(w, h);
		
		this->bNewCoordinate = false;
	}
}

void LighteningEffect::draw() const
{
}
