/*!
 * @file glgui_widget.h
 * The gl_widget of the openglGUI
 */

#ifndef _GLGUI_WIDGET_H
#define _GLGUI_WIDGET_H

#include "element_2d.h"

#include "glgui_style.h"
#include "material.h"
#include "rect2D.h"

#include "event.h"
#include "signal_connector.h"

namespace OrxGui
{

  class GLGuiCursor;


  //! This is widget part of the openglGUI class
  /**
   * A widget is the main class of all the elements of th GUI.
   */
  class GLGuiWidget : public Element2D
  {
  public:
    GLGuiWidget(GLGuiWidget* parent = NULL);
    virtual ~GLGuiWidget();

    void show();
    void hide();

    void setParentWidget(GLGuiWidget* parent);
    GLGuiWidget* parent() const { return this->_parent; }

    /// FOCUS
    /** @brief gives focus to this widget */
    void giveFocus();
    void breakFocus();
    /** @returns true if the widget is focusable */
    bool focusable() const { return this->_focusable; };
    /** @param focusable sets if the Widget should be focusable */
    void setFocusable(bool focusable = true) { this->_focusable = focusable; };
    /** @returns true if the position is inside of the Widget. @param position the position to check */
    bool focusOverWidget(const Vector2D& position) const;
    /** @brief overloaded function, that returns true if the cursor is on top of the Widget */
    bool focusOverWidget(const OrxGui::GLGuiCursor* const cursor) const;

    /** @returns the currently focused Widget (NULL if none is selected) */
    static GLGuiWidget* focused() { return GLGuiWidget::_focused; };


    /// CLICK
    void click(const Vector2D& pos);
    void release(const Vector2D& pos);
    bool clickable() const { return this->_clickable; };
    void setClickable(bool clickable = true) { this->_clickable = clickable; };

    static void connect(GLGuiWidget* sender, Signal& signal, BaseObject* receiver, Slot executor);
    void connect(Signal& signal, BaseObject* receiver, Slot executor);

    void disconnect(GLGuiWidget* sender, Signal& signal, BaseObject* receiver);


    /// MATERIAL (looks)
    Material& backMaterial() { return this->_backMat; };
    const Material& backMaterial() const { return this->_backMat; };
    Rect2D& backRect() { return this->_backRect; };
    const Rect2D& backRect() const { return this->_backRect; };

    void setFrontColor(const Color& frontColor, bool instantaniously = false);
    const Color& frontColor() const { return this->_frontColor; };

    /** @brief sets all borders to the same value. */
    void setBorderSize(float borderSize);
    void setBorderLeft(float borderLeft);
    void setBorderRight(float borderRight);
    void setBorderTop(float borderTop);
    void setBorderBottom(float borderBottom);

    float borderLeft() const { return this->_borderLeft; };
    float borderRight() const { return this->_borderRight; };
    float borderTop() const { return this->_borderTop; };
    float borderBottom() const { return this->_borderBottom; };


    void setWidgetSize(const Vector2D& size);
    void setWidgetSize(float x, float y);


    void setBackgroundColor(float x, float y, float z) { this->backMaterial().setDiffuse(x,y,z); };

    inline void drawRect(const Rect2D& rect) const
    {
      glBegin(GL_QUADS);
      glTexCoord2i(0,0); glVertex2d(rect.left(), rect.top());
      glTexCoord2i(0,1); glVertex2d(rect.left(), rect.bottom());
      glTexCoord2i(1,1); glVertex2d(rect.right(), rect.bottom());
      glTexCoord2i(1,0); glVertex2d(rect.right(), rect.top());
      glEnd();
    }


    virtual void update() {};
    virtual void tick(float dt);
    virtual void draw() const;

    /** @param the Event to process. @returns true if the Event has been consumed*/
    virtual bool processEvent(const Event& event) { return false; };


  protected:
    /// LOOKS
    virtual void resize();

    virtual void hiding() {};
    virtual void showing() {};
    virtual void updateFrontColor() {};

    inline void beginDraw() const { glPushMatrix(); glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0); };
    inline void endDraw() const { glPopMatrix(); };

    /// EVENTS
    // if something was clickt on the GUI-widget.
    virtual void clicking(const Vector2D& pos);
    virtual void releasing(const Vector2D& pos);
    virtual void receivedFocus();
    virtual void removedFocus();

    virtual void destroyed();

  private:
    void init();

  private:
    GLGuiWidget*                   _parent;           //!< The parent of this Widget.

    /// LOOKS
    Material                       _backMat;
    Rect2D                         _backRect;

    Color                          _frontColor;
    Color*                         _toFrontColor;

    float                          _borderLeft;
    float                          _borderRight;
    float                          _borderTop;
    float                          _borderBottom;

    Vector2D                       _minSize;

    GLGuiStyle                     _style;

    /// EVENTS
    bool                           _focusable;        //!< If this widget can receive focus.
    bool                           _clickable;        //!< if this widget can be clicked upon.

    bool                           _pushed;


    static GLGuiWidget*            _selected;         //!< The currently selected Widget.
    static GLGuiWidget*            _focused;          //!< The currently Focused Widget.

    static GLGuiWidget*            _inputGrabber;     //!< The Widget that grabs input.
  };
}
#endif /* _GLGUI_WIDGET_H */
