/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GUI

#include "glgui_widget.h"

#include "glgui_cursor.h"

#include "material.h"

#include "debug.h"

namespace OrxGui
{

  /**
   * @brief standard constructor
  */
  GLGuiWidget::GLGuiWidget (GLGuiWidget* parent)
  {
    this->init();

    this->setParentWidget(parent);
  }


  /**
   * @brief standard deconstructor
   */
  GLGuiWidget::~GLGuiWidget()
  {
    if (this == GLGuiWidget::_focused)
      GLGuiWidget::_focused = NULL;

    if (this->_toFrontColor)
      delete this->_toFrontColor;
  }

  GLGuiWidget* GLGuiWidget::_selected = NULL;
  GLGuiWidget* GLGuiWidget::_focused = NULL;
  GLGuiWidget* GLGuiWidget::_inputGrabber = NULL;



  /**
   * initializes the GUI-element
   */
  void GLGuiWidget::init()
  {
    this->setClassID(CL_GLGUI_WIDGET, "GLGuiWidget");

    this->_focusable = false;
    this->_clickable = false;
    this->_pushed = false;

    this->setVisibility(GLGUI_WIDGET_DEFAULT_VISIBLE);

    this->_backMat.setDiffuseColor(Color(1.0, 0.5, 0.4, 1.0));
    this->_backMat.setDiffuseMap("gui_element_background.png");
    this->_frontColor = Color(1.0, 0.0, 0.0);
    this->_toFrontColor = NULL;


    this->_borderLeft = 15.0;
    this->_borderRight = 1.0;
    this->_borderTop = 1.0;
    this->_borderBottom = 1.0;
  }


  void GLGuiWidget::setParentWidget(GLGuiWidget* parent)
  {
    this->_parent = parent;

    if (parent != NULL)
      parent->addChild2D(this);
  }

  /** @brief gives focus to this widget */
  void GLGuiWidget::giveFocus()
  {
    if (GLGuiWidget::focused() != NULL)
      GLGuiWidget::focused()->breakFocus();
    GLGuiWidget::_focused = this;
    this->receivedFocus();
  };

  void GLGuiWidget::breakFocus()
  {
    if (GLGuiWidget::_focused == this)
    {
      GLGuiWidget::_focused = NULL;
      this->_pushed = false;
      this->removedFocus();
    }
  };


  bool GLGuiWidget::focusOverWidget(const Vector2D& position) const
  {
    return (this->getAbsCoor2D().x < position.x && this->getAbsCoor2D().x + this->getSizeX2D() > position.x &&
            this->getAbsCoor2D().y < position.y && this->getAbsCoor2D().y + this->getSizeY2D() > position.y);
  }

  bool GLGuiWidget::focusOverWidget(const GLGuiCursor* const cursor) const
  {
    return this->focusOverWidget(cursor->getAbsCoor2D());
  }

  void GLGuiWidget::setFrontColor(const Color& frontColor, bool instantaniously)
  {
    if (instantaniously)
    {
      this->_frontColor = frontColor;
      if (this->_toFrontColor != NULL)
      {
        delete this->_toFrontColor;
        this->_toFrontColor = NULL;
      }
    }
    else if (!this->_toFrontColor)
      this->_toFrontColor = new Color(frontColor);
    else
      *this->_toFrontColor = frontColor;
    //this->_frontColor = frontColor;
    //this->updateFrontColor();
  };


  void GLGuiWidget::setBorderSize(float borderSize)
  {
    this->_borderLeft = borderSize;
    this->_borderRight = borderSize;
    this->_borderTop = borderSize;
    this->_borderBottom = borderSize;
    this->resize();
  }

  void GLGuiWidget::setBorderLeft(float borderLeft)
  {
    this->_borderLeft = borderLeft;
    this->resize();
  }
  void GLGuiWidget::setBorderRight(float borderRight)
  {
    this->_borderRight = borderRight;
    this->resize();
  }
  void GLGuiWidget::setBorderTop(float borderTop)
  {
    this->_borderTop = borderTop;
    this->resize();
  }
  void GLGuiWidget::setBorderBottom(float borderBottom)
  {
    this->_borderBottom = borderBottom;
    this->resize();
  }



  void GLGuiWidget::resize()
  {
    this->backRect().setSize(this->getSize2D());
    if (this->parent() != NULL)
      this->parent()->resize();
  }


  void GLGuiWidget::click(const Vector2D& pos)
  {
    assert (!this->_pushed);
    this->_pushed = true;

    this->clicking(pos);
  }

  void GLGuiWidget::release(const Vector2D& pos)
  {
    if (this->_pushed)
    {
      this->releasing(pos);
      this->_pushed = false;
    }
  }


  void GLGuiWidget::clicking(const Vector2D& pos)
  {
    this->setFrontColor(Color(0, 0, 1));

  }

  void GLGuiWidget::releasing(const Vector2D& pos)
  {
    this->setFrontColor(Color(0,1,0));

  }

  void GLGuiWidget::receivedFocus()
  {
    this->setFrontColor(Color(0, 1, 0));
  }

  void GLGuiWidget::removedFocus()
  {
    this->setFrontColor(Color(1, 0, 0));

  }

  void GLGuiWidget::destroyed()
  {}
  ;


  void GLGuiWidget::setWidgetSize(const Vector2D& size)
  {
    this->setSize2D(size);
    this->resize();

  }


  void GLGuiWidget::setWidgetSize(float x, float y)
  {
    this->setWidgetSize(Vector2D(x, y));
  }



  void GLGuiWidget::connect(GLGuiWidget* sender, Signal& signal, BaseObject* receiver, Slot executor)
  {
    sender->connect(signal, receiver, executor);
  }

  void GLGuiWidget::connect(Signal& signal, BaseObject* receiver, Slot executor)
  {
    signal.push_back(SignalConnector(receiver, executor));
  }


  void GLGuiWidget::show()
  {
    this->setVisibility(true);
    this->showing();
  }



  void GLGuiWidget::hide()
  {
    this->setVisibility(false);
    this->hiding();
  }

  void GLGuiWidget::tick(float dt)
  {
    if (this->_toFrontColor)
    {
      this->_frontColor.slerp(*_toFrontColor, dt*3.0);
      this->updateFrontColor();
      if (this->_frontColor.dist(*_toFrontColor) < .1)
      {
        delete _toFrontColor;
        _toFrontColor = NULL;
      }
    }
  }


  /**
   * USE THIS FUNCTION ONLY FROM DERIVED CLASS
   */
  void GLGuiWidget::draw() const
  {
    this->backMaterial().select();
    this->drawRect(this->backRect());
    this->backMaterial().unselect();
  }

}
