/*!
 * @file material.h
 * @brief Contains the Material Class that handles Material for 3D-Objects.
 * @todo free SDL-surface when deleting Material.
 * @todo delete imgNameWithPath after use creation.
 */

#ifndef _MATERIAL_H
#define _MATERIAL_H
#include "base_object.h"


#if HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <vector>
#include "texture.h"
#include "color.h"

// FORWARD DECLARATIONS //

//! Class to handle Materials.
class Material : public BaseObject
{
  public:
    Material (const std::string& mtlName = "");
    virtual ~Material ();

    Material& operator=(const Material& material);

    bool select () const;
    bool activateTextureUnit(unsigned int textureNumber);
    static void unselect();

    void setIllum (int illum);
    int getIllumModel() const { return this->illumModel; };

    void setDiffuse (float r, float g, float b);
    void setAmbient (float r, float g, float b);
    void setSpecular (float r, float g, float b);
    void setShininess (float shini);
    void setTransparency (float trans);
    void setBlendFunc(GLenum sFactor, GLenum tFactor) { this->sFactor = sFactor; this->tFactor = tFactor; };

    const Color& getDiffuseColor() const { return diffuse; };

    // MAPPING //
    void setDiffuseMap(const Texture& texture, unsigned int textureNumber = 0);
    void setDiffuseMap(const std::string& dMap, GLenum target = GL_TEXTURE_2D, unsigned int textureNumber = 0);
    void setSDLDiffuseMap(SDL_Surface *surface, GLenum target = GL_TEXTURE_2D, unsigned int textureNumber = 0);
    void renderToTexture(unsigned int textureNumber, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height);

    void setAmbientMap(const std::string& aMap, GLenum target = GL_TEXTURE_2D);
    void setSpecularMap(const std::string& sMap, GLenum target = GL_TEXTURE_2D);
    void setBump(const std::string& bump);
    GLuint getDiffuseTexture(unsigned int i = 0) const { return (this->textures.size() > i)? this->textures[i].getTexture() : 0; };

    static void addTexturePath(const std::string& pathName);

  public:
    static const GLenum glTextureArbs[];  //!< The Texture ARB's

    static unsigned int getMaxTextureUnits();

  private:
    static const Material* selectedMaterial; //!< The currently selected material.

    int              illumModel;       //!< The IlluminationModel is either flat or smooth.
    Color            diffuse;          //!< The diffuse color of the Material. (also transparency.)
    Color            ambient;          //!< The ambient color of the Material.
    Color            specular;         //!< The specular color of the Material.
    float            shininess;        //!< The shininess of the Material.
    GLenum           sFactor;          //!< The Blending Factor for the Source.
    GLenum           tFactor;          //!< The Blending Factor for the Destination.

    std::vector<Texture> textures;     //!< An Array of Textures.

    Texture*         ambientTexture;   //!< The ambient texture of the Material.
    Texture*         specularTexture;  //!< The specular texture of the Material.
};

#endif
