/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christoph Renner
   co-programmer:
*/

#include "udp_socket.h"
#include "udp_server_socket.h"
#include "debug.h"


void UdpSocket::init( )
{
  //TODO setClassId
  this->serverSocket = NULL;
  this->socket = NULL;
  this->packet = NULL;
}


/**
 * constructor - connects to give host
 * @param host host
 * @param port port
 */
UdpSocket::UdpSocket( std::string host, int port )
{
  init();
  this->packet = SDLNet_AllocPacket( UDP_PACKET_SIZE );

  assert( this->packet );

  memset( packet->data, 0, UDP_PACKET_SIZE );
  PRINTF(0)("PACKET DATA: %x\n", packet->data);

  this->connectToServer( host, port );
}

/**
 * default constructor. use this one if you want to call connecttoServer
 */
UdpSocket::UdpSocket( )
{
  this->init();
  this->packet = SDLNet_AllocPacket( UDP_PACKET_SIZE );

  if ( !packet )
  {
    PRINTF(1)("SDLNet_AllocPacket: %s\n", SDLNet_GetError());

    assert( false );
    bOk = false;
  }

}

/**
 * constructor. used by UdpServerSocket
 * @param serverSocket pointer to serverSocket
 * @param ip client's ip address
 * @param userId userid used by serverSocket
 */
UdpSocket::UdpSocket( UdpServerSocket * serverSocket, IPaddress ip, int userId )
{
  this->init();
  this->serverSocket = serverSocket;
  this->userId = userId;
}

/**
 * destructor
 */
UdpSocket::~UdpSocket( )
{
  this->disconnectServer();

  if ( serverSocket )
    serverSocket->removeUser( userId );

  if ( this->packet )
    SDLNet_FreePacket( this->packet );

  if ( socket )
    SDLNet_UDP_Close( socket );
}

/**
 * connect to server
 * @param host host name
 * @param port port number
 */
void UdpSocket::connectToServer( std::string host, int port )
{
  assert( serverSocket == NULL );

  IPaddress ip;

  PRINTF(0)("connect to server %s on port %d\n", host.c_str(), port);

  if ( SDLNet_ResolveHost( &ip, host.c_str(), port ) != 0 )
  {
    PRINTF(1)("SDLNet_ResolveHost: %s\n", SDLNet_GetError() );
    bOk = false;
    return;
  }

  socket = SDLNet_UDP_Open(0);
  if ( !socket )
  {
    PRINTF(1)("SDLNet_UDP_Open: %s\n", SDLNet_GetError() );
    bOk = false;
    return;
  }

  int channel = SDLNet_UDP_Bind(socket, 1, &ip);
  if ( channel == -1 )
  {
    PRINTF(1)("SDLNet_UDP_Bind: %s\n", SDLNet_GetError());
    bOk = false;
    return;
  }
}

/**
 * disconnect from server
 */
void UdpSocket::disconnectServer( )
{
  PRINTF(0)("disconnect\n");
  writePacket( NULL, 0 );
  SDLNet_UDP_Unbind( socket, -1 );
  SDLNet_UDP_Close( socket );
  bOk = false;
  socket = NULL;
}

/**
 * send one packet to other host
 * @param data pointer to data which will be sent
 * @param length length of data
 * @return true on success
 */
bool UdpSocket::writePacket( byte * data, int length )
{
  if ( serverSocket )
  {
    NetworkPacket networkPacket;
    networkPacket.length = length;
    networkPacket.data = data;
    if ( !serverSocket->sendPacket( networkPacket, this->userId ) )
    {
      bOk = false;
      return false;
    }
    else
      return true;
  }
  else
  {
    assert( length <= packet->maxlen );

    memcpy( packet->data, data, length );
    packet->len = length;

    if ( socket && SDLNet_UDP_Send( socket, 1, packet) == 0 )
    {
      PRINTF(1)("SDLNet_UDP_Send: %s\n", SDLNet_GetError());
      bOk = false;
      return false;
    }

    return true;
  }
}

/**
 * recieve one packet from another host
 * @param data pointer to buffer to copy data into
 * @param maxLength maximal length of buffer
 * @return less than 0 on error, number bytes read else
 */
int UdpSocket::readPacket( byte * data, int maxLength )
{
  assert( maxLength <= UDP_PACKET_SIZE );

  if ( serverSocket )
  {
    NetworkPacket networkPacket = serverSocket->getPacket( this->userId );

    if ( networkPacket.length == -1 )
    {
      this->disconnectServer();
      return 0;
    }

    if ( networkPacket.length > 0 )
    {
      assert( maxLength > networkPacket.length );

      memcpy( data, networkPacket.data, networkPacket.length );
    }

    if ( networkPacket.data )
    {
      free( networkPacket.data );
      networkPacket.data = NULL;
    }

    return networkPacket.length;
  }
  else
  {
    int numrecv = SDLNet_UDP_Recv( socket, packet);
    if ( numrecv > 0)
    {
      assert( packet->len <= maxLength );

      if ( packet->len == 0 )
      {
        this->disconnectServer();
        return 0;
      }

      memcpy( data, packet->data, packet->len );
      return packet->len;
    }
    else if ( numrecv < 0 )
    {
      PRINTF(1)("SDLNet_UDP_Recv: %s\n", SDLNet_GetError());
      bOk = false;
      return -1;
    }
    else
    {
      return 0;
    }
  }

  return 0;
}



