/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "material.h"

#include "texture.h"
#include "debug.h"
#include "compiler.h"
#include "util/loading/resource_manager.h"

/**
 * @brief creates a Material.
 * @param mtlName Name of the Material to be added to the Material List
 */
Material::Material (const std::string& mtlName)
{
  this->setClassID(CL_MATERIAL, "Material");

  this->setIllum(3);
  this->setDiffuse(1,1,1);
  this->setAmbient(0,0,0);
  this->setSpecular(.5,.5,.5);
  this->setShininess(2.0);
  this->setTransparency(1.0);

  this->ambientTexture = NULL;
  this->specularTexture = NULL;
  this->sFactor = GL_SRC_ALPHA;
  this->tFactor = GL_ONE;

  this->setName(mtlName);
}

/**
 * @brief deletes a Material
 */
Material::~Material()
{
  PRINTF(5)("delete Material %s.\n", this->getName());

  if (this->ambientTexture != NULL)
    ResourceManager::getInstance()->unload(this->ambientTexture);
  if (this->specularTexture != NULL)
    ResourceManager::getInstance()->unload(this->specularTexture);

  if (this == Material::selectedMaterial)
    Material::selectedMaterial = NULL;
}


const Material* Material::selectedMaterial = NULL;

const GLenum Material::glTextureArbs[] =
{
  GL_TEXTURE0,
  GL_TEXTURE1,
  GL_TEXTURE2,
  GL_TEXTURE3,
  GL_TEXTURE4,
  GL_TEXTURE5,
  GL_TEXTURE6,
  GL_TEXTURE7
};


/// TODO FIX THIS
// Material& Material::operator=(const Material& m)
// {
//   this->setIllum(m.illumModel);
//   this->setDiffuse(m.diffuse[0],m.diffuse[1],m.diffuse[2]);
//   this->setAmbient(m.ambient[0],m.ambient[1],m.ambient[2]);
//   this->setSpecular(m.specular[0],m.specular[1],m.specular[2]);
//   this->setShininess(m.shininess);
//   this->setTransparency(m.transparency);
//
//   if (this->diffuseTexture != NULL)
//     ResourceManager::getInstance()->unload(this->diffuseTexture);
//   if (m.diffuseTexture != NULL)
//     this->diffuseTexture = (Texture*)ResourceManager::getInstance()->copy(m.diffuseTexture);
//   this->ambientTexture = NULL; /// FIXME
//   this->specularTexture = NULL; /// FIXME
//
//   this->setName(m.getName());
// }



/**
 * @brief sets the material with which the following Faces will be painted
 */
  bool Material::select() const
{
  if (unlikely(this == Material::selectedMaterial))
      return true;

/// !!  HACK !!! FIX THIS IN MODEL ///
  else if (likely(Material::selectedMaterial != NULL))
  {
  Material::unselect();
//     for(unsigned int i = 0; i < Material::selectedMaterial->textures.size(); ++i)
//     {
//         glActiveTexture(Material::glTextureArbs[i]);
//         glBindTexture(GL_TEXTURE_2D, 0);
//         glDisable(GL_TEXTURE_2D);
//     }
  }

  if (likely(Material::selectedMaterial != NULL))
  {
    for(unsigned int i = 0; i < Material::selectedMaterial->textures.size(); ++i)
    {
        glActiveTexture(Material::glTextureArbs[i]);
        //glBindTexture(GL_TEXTURE_2D, 0);
        glDisable(GL_TEXTURE_2D);
    }
  }

    // setting diffuse color
  glColor4f (diffuse[0], diffuse[1], diffuse[2], this->transparency);
  // setting ambient color
  glMaterialfv(GL_FRONT, GL_AMBIENT, this->ambient);
  // setting up Sprecular
  glMaterialfv(GL_FRONT, GL_SPECULAR, this->specular);
  // setting up Shininess
  glMaterialf(GL_FRONT, GL_SHININESS, this->shininess);

  // setting the transparency
  if (this->transparency < 1.0 ||       /* This allows alpha blending of 2D textures with the scene */
      (likely(!this->textures.empty() && this->textures[0].hasAlpha())))
  {
    glEnable(GL_BLEND);
    glBlendFunc(this->sFactor, this->tFactor);
  }
  else
  {
    glDisable(GL_BLEND);
  }


  // setting illumination Model
  if (this->illumModel == 1) //! @todo make this work, if no vertex-normals are read.
    glShadeModel(GL_FLAT);
  else if (this->illumModel >= 2)
    glShadeModel(GL_SMOOTH);

  for(unsigned int i = 0; i < this->textures.size(); ++i)
  {
      glActiveTexture(Material::glTextureArbs[i]);
      glEnable(GL_TEXTURE_2D);
      if(this->textures[i].hasAlpha())
      {
        glEnable(GL_BLEND);
      }
      glBindTexture(GL_TEXTURE_2D, this->textures[i].getTexture());
  }
  Material::selectedMaterial = this;

  return true;
}

void Material::unselect()
{
  Material::selectedMaterial = NULL;
    for(unsigned int i = 0; i < 8; ++i)
    {
        glActiveTexture(Material::glTextureArbs[i]);
        glBindTexture(GL_TEXTURE_2D, 0);
        glDisable(GL_TEXTURE_2D);
    }
}

/**
 *  Sets the Material Illumination Model.
 *  illu illumination Model in int form
 */
void Material::setIllum (int illum)
{
  PRINTF(4)("setting illumModel of Material %s to %i\n", this->getName(), illum);
  this->illumModel = illum;
}

/**
 *  Sets the Material Illumination Model.
 *  illu illumination Model in char* form
 */
void Material::setIllum (char* illum)
{
  this->setIllum (atoi(illum));
}

/**
 *  Sets the Material Diffuse Color.
 * @param r Red Color Channel.a
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setDiffuse (float r, float g, float b)
{
  PRINTF(4)("setting Diffuse Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->diffuse[0] = r;
  this->diffuse[1] = g;
  this->diffuse[2] = b;
  this->diffuse[3] = 1.0;

}

/**
 *  Sets the Material Diffuse Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
 */
void Material::setDiffuse (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setDiffuse (r, g, b);
}

/**
 *  Sets the Material Ambient Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
*/
void Material::setAmbient (float r, float g, float b)
{
  PRINTF(4)("setting Ambient Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->ambient[0] = r;
  this->ambient[1] = g;
  this->ambient[2] = b;
  this->ambient[3] = 1.0;
}

/**
 *  Sets the Material Ambient Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
 */
void Material::setAmbient (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setAmbient (r, g, b);
}

/**
 *  Sets the Material Specular Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setSpecular (float r, float g, float b)
{
  PRINTF(4)("setting Specular Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->specular[0] = r;
  this->specular[1] = g;
  this->specular[2] = b;
  this->specular[3] = 1.0;
}

/**
 *  Sets the Material Specular Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setSpecular (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setSpecular (r, g, b);
}

/**
 *  Sets the Material Shininess.
 * @param shini stes the Shininess from float.
*/
void Material::setShininess (float shini)
{
  this->shininess = shini;
}
/**
 *  Sets the Material Shininess.
 * @param shini stes the Shininess from char*.
*/
void Material::setShininess (char* shini)
{
  this->setShininess (atof(shini));
}

/**
 *  Sets the Material Transparency.
 * @param trans stes the Transparency from int.
*/
void Material::setTransparency (float trans)
{
  PRINTF(4)("setting Transparency of Material %s to %f.\n", this->getName(), trans);
  this->transparency = trans;
}
/**
 *  Sets the Material Transparency.
 * @param trans stes the Transparency from char*.
*/
void Material::setTransparency (char* trans)
{
  this->setTransparency (atof(trans));
}

/**
 *  Adds a Texture Path to the List of already existing Paths
 * @param pathName The Path to add.
*/
void Material::addTexturePath(const std::string& pathName)
{
  ResourceManager::getInstance()->addImageDir(pathName);
}

// MAPPING //

void Material::setDiffuseMap(const Texture& texture, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  //! @todo check if RESOURCE MANAGER is availiable
  this->textures[textureNumber] = texture;
}


/**
 * @brief Sets the Materials Diffuse Map
 * @param dMap the Name of the Image to Use
*/
void Material::setDiffuseMap(const std::string& dMap, GLenum target, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  PRINTF(5)("setting Diffuse Map %s\n", dMap.c_str());
  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  //! @todo check if RESOURCE MANAGER is availiable
  if (!dMap.empty())
  {
    Texture* tex = dynamic_cast<Texture*>(ResourceManager::getInstance()->load(dMap, IMAGE, RP_GAME, (int)target));
    if (tex != NULL)
    this->textures[textureNumber] = *tex;
    else
      this->textures[textureNumber] = Texture();
  }
  else
  {
    this->textures[textureNumber] = Texture();
  }
}

/**
 * @brief Sets the Materials Diffuse Map
 * @param surface pointer to SDL_Surface which shall be used as texture
*/
void Material::setSDLDiffuseMap(SDL_Surface *surface, GLenum target, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());


  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, Texture());

  if(surface != NULL)
  {
    this->textures[textureNumber] = Texture(surface, GL_TEXTURE_2D);
  }
  else
  {
    this->textures[textureNumber] = Texture();
  }

}

/**
 * @brief renders viewport buffer (?? or another buffer ;-)) to a texture
 * @param textureNumber select the texture unit that will be overwritten
*/
void Material::renderToTexture(unsigned int textureNumber, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)
{
  assert(textureNumber < Material::getMaxTextureUnits());
  assert(textureNumber < this->textures.size());

  // HACK
  glActiveTexture(textureNumber);
   glEnable(GL_TEXTURE_2D);
   glBindTexture(GL_TEXTURE_2D, this->textures[textureNumber].getTexture());
  glCopyTexSubImage2D(target, level, xoffset, yoffset, x, y, width, height);

}

/**
 * @brief Sets the Materials Ambient Map
 * @todo implement this
*/
void Material::setAmbientMap(const std::string& aMap, GLenum target)
{
  /// FIXME SDL_Surface* ambientMap;

}

/**
 * @brief Sets the Materials Specular Map
 * @param sMap the Name of the Image to Use
   @todo implement this
*/
void Material::setSpecularMap(const std::string& sMap, GLenum target)
{
  /// FIXME SDL_Surface* specularMap;

}

/**
 * @brief Sets the Materials Bumpiness
 * @param bump the Name of the Image to Use
 * @todo implemet this
*/
void Material::setBump(const std::string& bump)
{
}



unsigned int Material::getMaxTextureUnits()
{
  int maxTexUnits = 0;
  glGetIntegerv(GL_MAX_TEXTURE_UNITS, &maxTexUnits);
  return maxTexUnits;
}
