/*
	orxonox - the future of 3D-vertical-scrollers

	Copyright (C) 2004 orx

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

### File Specific:
	main-programmer: hdavid, amaechler
*/

#include "fog_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "shell_command.h"

SHELL_COMMAND(activate, FogEffect, activateFog);
SHELL_COMMAND(deactivate, FogEffect, deactivateFog);
SHELL_COMMAND(startfogging, FogEffect, startFogging);

using namespace std;

CREATE_FACTORY(FogEffect, CL_FOG_EFFECT);

FogEffect::FogEffect(const TiXmlElement* root)
{
	this->setClassID(CL_FOG_EFFECT, "FogEffect");

	this->init();

	if (root != NULL)
		this->loadParams(root);

	if (this->fogActivate)
		this->activate();
}


FogEffect::~FogEffect()
{
	this->deactivate();
}


void FogEffect::loadParams(const TiXmlElement* root)
{
	WeatherEffect::loadParams(root);

	LoadParam(root, "mode", this, FogEffect, setFogMode);
	LoadParam(root, "density", this, FogEffect, setFogDensity);
	LoadParam(root, "range", this, FogEffect, setFogRange);
	LoadParam(root, "color", this, FogEffect, setFogColor);

	LOAD_PARAM_START_CYCLE(root, element);
	{
		LoadParam_CYCLE(element, "option", this, FogEffect, setFogOption);
	}
	LOAD_PARAM_END_CYCLE(element);
}

bool FogEffect::init()
{
	//default values
	this->fogActivate = false;
	this->fogFadeDuration = 0;
	this->localTimer = 0;

	this->fogMode = GL_EXP2;
	this->fogDensity = 0.001;
	this->fogStart = 0;
	this->fogEnd = 5000;
	this->colorVector = Vector(0.7, 0.7, 0.7);

        return true;
}


bool FogEffect::activate()
{
	PRINTF(0)( "Enabling FogEffect, mode: %i, density: %f, start: %f, end: %f, color %f, %f, %f\n", this->fogMode, this->fogDensity, this->fogStart, this->fogEnd, this->colorVector.x, this->colorVector.y, this->colorVector.z);

	this->fogActivate = true;

	GLfloat fogColor[4] = { colorVector.x, colorVector.y, colorVector.z, 1.0};
	glFogi(GL_FOG_MODE, this->fogMode);
	glFogfv(GL_FOG_COLOR, fogColor);
	glFogf(GL_FOG_DENSITY, this->fogDensity);
	glHint(GL_FOG_HINT, GL_DONT_CARE);
	glFogf(GL_FOG_START, this->fogStart);
	glFogf(GL_FOG_END, this->fogEnd);

	glEnable(GL_FOG);
	// glClearColor(0.5, 0.5, 0.5, 1.0);
        return true;
}


bool FogEffect::deactivate()
{
	PRINTF(0)("Deactivating FogEffect\n");

	this->fogActivate = false;

	glDisable(GL_FOG);

        return true;
}

void FogEffect::draw() const {

	if (this->fogFadeDuration != 0 && this->localTimer < this->fogFadeDuration)
		glFogf(GL_FOG_DENSITY, this->fogFadeDensity);
	//else
	//	glFogf(GL_FOG_DENSITY, this->fogDensity);

}
void FogEffect::tick(float dt)
{
	if (!this->fogActivate)
		return;


	if (this->fogFadeDuration != 0 && this->localTimer < this->fogFadeDuration) {
		this->localTimer += dt;
		float progress = this->localTimer / this->fogFadeDuration;
		this->fogFadeDensity = progress * this->fogDensity;
	}
}

void FogEffect::startFogging() {

	if (this->fogActivate)
		this->deactivate();

	this->fogFadeDuration = 10;
	this->localTimer = 0;
	this->activate();

}


GLint FogEffect::stringToFogMode(const std::string& mode)
{
	if(mode == "GL_LINEAR")
		return GL_LINEAR;
	else if(mode == "GL_EXP")
		return GL_EXP;
	else if(mode == "GL_EXP2" )
		return GL_EXP2;
	else
		return -1;
}


