/*!
 * @file element_2d.h
 * Definition of the 2D elements rendered on top of all other stuff.
 */

#ifndef _ELEMENT_2D_H
#define _ELEMENT_2D_H

#include "base_object.h"

#include "vector.h"
#include "vector2D.h"
#include <list>

// FORWARD DECLARATION
class PNode;
class TiXmlElement;

//!< An enumerator defining the Depth of a 2D-element.
typedef enum
{
  E2D_LAYER_BELOW_ALL           =     0,        //!< Will be rendered below the 3D-scene. @todo make this work.
  E2D_LAYER_BOTTOM              =     1,        //!< Will be rendered on the bottom Layer
  E2D_LAYER_MEDIUM              =     2,        //!< Will be rendered on the medium Layer.
  E2D_LAYER_TOP                 =     3,        //!< Will be rendered on top of everything else
  E2D_LAYER_ABOVE_ALL           =     4,        //!< Will be rendered above everything else.

  E2D_LAYER_COUNT               =     5,         //!< The count of Layers.
} E2D_LAYER;
#define E2D_DEFAULT_LAYER       E2D_LAYER_MEDIUM
#define E2D_LAYER_ALL           5

typedef enum
{
  E2D_ALIGN_NONE                =     0,
  E2D_ALIGN_LEFT                =     1,
  E2D_ALIGN_RIGHT               =     2,
  E2D_ALIGN_CENTER              =     4,
  E2D_ALIGN_SCREEN_CENTER       =     8
} E2D_ALIGNMENT;

typedef enum
{
  E2D_PARENT_NONE                    = 0x0000,
  E2D_PARENT_LOCAL_ROTATE            = 0x0001,    //!< Rotates all the children around their centers.
  E2D_PARENT_ROTATE_MOVEMENT         = 0x0002,    //!< Moves all the children around the center of their parent, without the rotation around their own centers.

  E2D_PARENT_MOVEMENT                = 0x0004,    //!< Moves all children along with the parent.
// special linkage modes
  E2D_PARENT_ALL                     = 0x0003,    //!< Moves all children around the center of their parent, and also rotates their centers
  E2D_PARENT_ROTATE_AND_MOVE         = 0x0005,    //!< Rotates all children around their axis, and moves them as the Parent Moves, but does not rotate around the center of their parent.


  // REPARENTING
  E2D_REPARENT_TO_NULL               = 0x0010,    //!< Reparents to the Null, if the Parent is Removed. Meaning the Node wont have a parent anymore.
  E2D_REPARENT_TO_PARENTS_PARENT     = 0x0020,    //!< Reparents the Node to the parents (old) parent it the parent gets removed.
  /////////////////////////////////////////////   //  ELSE: Reparents to the NullParent.
  E2D_REPARENT_DELETE_CHILDREN       = 0x0040,    //!< Deletes the Children of the node when This Node is Removed. (Use with care).
  /// FIXME
   E2D_REPARENT_KEEP_POSITION         = 0x0080,   //!< Tries to keep the Position if the Node is reparented.


  // DELETION
  E2D_PROHIBIT_CHILD_DELETE          = 0x0100,    //!< Prohibits the Children from being deleted if this Node gets deleted.
  E2D_PROHIBIT_DELETE_WITH_PARENT    = 0x0200,    //!< Prohibits the Node to be deleted if the Parent is. Child will be reparented according to the Repaenting-Rules
  E2D_REPARENT_CHILDREN_ON_REMOVE    = 0x0400,    //!< Reparents the Children of the Node if the Node gets Removed.
  E2D_REPARENT_ON_PARENTS_REMOVE     = 0x0800,    //!< The Node gets Reparented if its Parent gets removed. Child will be reparented according to the Reparenting-Rules.

  // VISIBILITY/ACTIVITY
  E2D_HIDE_CHILDREN_IF_HIDDEN        = 0x1000,    //!< Prohibits the Children from being drawn if this node isn't visible. (used for Draw))
  //E2D_HIDE_IF_PARENT_HIDDEN          = 0x2000,    //!< Prohibits the node from being drawn if the Parent is invisible.
  E2D_UPDATE_CHILDREN_IF_INACTIVE    = 0x4000,    //!< Updates the Children of this Node even if the Parent is Inactive (note if this's parent is inactive children won't be updated.)
  E2D_STATIC_NODE                    = 0x8000,    //!< Used for nodes that do not have any moving children, and that do not move.

} E2D_PARENT_MODE;
#define     E2D_PARENT_MODE_DEFAULT     E2D_PARENT_ALL | \
                                        E2D_REPARENT_KEEP_POSITION

//! A class for 2D-elements
/**
 * this class mainly tries to imitate PNode in its functionality, but on a 2D-level
 * it extends PNode in the Sense of the tick-ability/draw-alility layering (and size)
 *
 * Layering: Layers are sorted into the Tree. e.g:
 * the roor is in the lowest Layer, the leaves in the highest (of each branche)
 * the first child of each node is in the lowest layer of all children, the last in the highest
 * -> the tree is sorted on insertion of a new Child: @see Element2D::addChild2D()
 */
class Element2D : virtual public BaseObject {

  public:
    Element2D(Element2D* parent = Element2D::getNullElement(), E2D_LAYER layer = E2D_DEFAULT_LAYER, short nodeFlags = E2D_PARENT_MODE_DEFAULT);
    virtual ~Element2D();

    virtual void loadParams(const TiXmlElement* root);

    // ACTIVATION //
    inline void activate2D() { this->bActive = this->bRelCoorChanged = this->bRelDirChanged = true; };
    inline void deactivate2D() { this->bActive = false; };
    inline bool get2DActiveState() { return this->bActive; };

    // ALIGNMENT //
    /** @param alignment the new Alignment of the 2D-Element */
    inline void setAlignment(E2D_ALIGNMENT alignment) { this->alignment = alignment; };
    void setAlignment(const std::string& alignment);
    inline E2D_ALIGNMENT getAlignment() const { return this->alignment; };

    // LAYERING //
    void setLayer(E2D_LAYER layer);
    void setLayer(const std::string& layer);
    /** @returns the Layer this Element is drawn to */
    inline E2D_LAYER getLayer() const { return this->layer; };

    // VISIBILITY //
    /** @param visible true if the Element should be visible false otherwise (will not be rendered) */
    inline void setVisibility(bool visible) { this->bVisible = visible; };
    /** @returns the visibility state */
    inline bool isVisible() const { return (this->bVisible && this->bCurrentlyVisible);  };


    // POSITIONAL (E2D-specials) //
    /** @param bindNode the Node this 2D-element should follow. if NULL the Element will not follow anything */
    void setBindNode(const PNode* bindNode);
    void setBindNode(const std::string& bindNode);
    inline const PNode* getBindNode() const { return this->bindNode; };

    inline void setSize2D(float x, float y) { this->size = Vector2D(x, y); };
    inline void setSize2D(const Vector2D& size) { this->size = size; };
    inline const Vector2D& getSize2D() const { return this->size; };
    void setSizeSoft2D(float x, float y, float bias = 1.0);
    inline void setSizeX2D(float x) { this->size.x = x; };
    inline void setSizeY2D(float y) { this->size.y = y; };
    inline float getSizeX2D() const { return this->size.x; };
    inline float getSizeY2D() const { return this->size.y; };

  public:
    virtual void tick(float dt) {};
    virtual void draw() const  {};
    void tick2D(float dt);
    void draw2D(E2D_LAYER from, E2D_LAYER to) const;
    void drawChildren(E2D_LAYER from, E2D_LAYER to) const;

    // LIKE PNODE
  public:
    void setRelCoor2D (const Vector2D& relCoord);
    void setRelCoorX2D(float x);
    void setRelCoorY2D(float y);
    void setRelCoor2D (float x, float y);
    void setRelCoor2Dpx (int x, int y);
    void setRelCoorSoft2D (const Vector2D& relCoordSoft, float bias = 1.0);
    void setRelCoorSoft2D (float x, float y, float bias = 1.0);
    void setRelCoorSoft2Dpx (int x, int y, float bias = 1.0);
    /** @returns the relative position */
    inline const Vector2D& getRelCoor2D () const { return this->prevRelCoordinate; };
    /** @returns the Relative Coordinate Destination */
    inline const Vector2D& getRelCoorSoft2D() const { return (this->toCoordinate)?*this->toCoordinate:this->relCoordinate; };
    const Vector2D& getRelCoor2Dpx() const;
    void setAbsCoor2D (const Vector2D& absCoord);
    void setAbsCoor2D (float x, float y);
    void setAbsCoorX2D(float x);
    void setAbsCoorY2D(float y);
    void setAbsCoor2Dpx (int x, int y);
    void setAbsCoorSoft2D (const Vector2D& absCoordSoft, float bias = 1.0);
    void setAbsCoorSoft2D (float x, float y, float bias = 1.0);
    /** @returns the absolute position */
    inline const Vector2D& getAbsCoor2D () const { return this->absCoordinate; };
    const Vector2D& getAbsCoor2Dpx () const;

    void shiftCoor2D (const Vector2D& shift);
    void shiftCoor2Dpx (int x, int y);

    void setRelDir2D (float relDir);
    void setRelDirSoft2D(float relDirSoft, float bias = 1.0);
    /** @returns the relative Direction */
    inline float getRelDir2D () const { return this->prevRelDirection; };
    /** @returns the Relative Directional Destination */
    inline float getRelDirSoft2D() const { return (this->toDirection)?*this->toDirection:this->relDirection; };
    void setAbsDir2D (float absDir);
    void setAbsDirSoft2D (float absDirSoft, float bias = 1.0);
    /** @returns the absolute Direction */
    inline float getAbsDir2D () const { return this->absDirection; };
    void shiftDir2D (float shiftDir);

    /** @returns the Speed of the Node */
    inline float getSpeed() const { return 0; };
    /** @returns the Velocity of the Node */
    inline const Vector2D& getVelocity() const { return this->velocity; };


    void addChild2D (Element2D* child);
    void addChild2D (const std::string& childName);
    void removeChild2D (Element2D* child);
    void remove2D();

    /** @param parent the new parent of this Element2D */
    void setParent2D (Element2D* parent) { parent->addChild2D(this); };
    void setParent2D (const std::string& parentName);
    /** @returns the parent of this Element2D */
    inline Element2D* getParent2D () const { return this->parent; };
    /** @returns the List of Children of this Element2D */
    inline const std::list<Element2D*>& getChildren2D() const { return this->children; };

    void setParentSoft2D(Element2D* parentNode, float bias = 1.0);
    void setParentSoft2D(const std::string& parentName, float bias = 1.0);

    void setParentMode2D (E2D_PARENT_MODE parentMode);
    void setParentMode2D (const std::string& parentingMode);
    /** @returns the Parenting mode of this node */
    int getParentMode2D() const { return this->parentMode; };

    // NULL_PARENT //
    /** @returns the NullParent, the (main) ROOT of the PNode Tree. If it does not yet exist, it will be created. */
    static Element2D* getNullElement()  { return (Element2D::nullElement != NULL)? Element2D::nullElement : Element2D::createNullElement(); };


    void update2D (float dt);

    void debug2D (unsigned int depth = 1, unsigned int level = 0) const;
    void debugDraw2D(unsigned int depth = 1, float size = 1.0, Vector color = Vector(1,0,0), unsigned int level = 0) const;

    // helper functions //
    static const char* parentingModeToString2D(int parentingMode);
    static E2D_PARENT_MODE stringToParentingMode2D(const std::string& parentingMode);

    static const char* layer2DToChar(E2D_LAYER layer);
    static E2D_LAYER charToLayer2D(const std::string& layer);

    static bool layerSortPredicate(const Element2D* elem1, const Element2D* elem2);

  private:
    void eraseChild2D(Element2D* child);
    /** tells the child that the parent's Coordinate has changed */
    inline void parentCoorChanged2D () { this->bRelCoorChanged = true; }
    /** tells the child that the parent's Direction has changed */
    inline void parentDirChanged2D () { this->bRelDirChanged = true; }
    /** @returns the last calculated coordinate */
    inline Vector2D getLastAbsCoor2D() { return this->lastAbsCoordinate; }

    void reparent2D();
    static Element2D* createNullElement();
    bool checkIntegrity(const Element2D* checkParent) const;


  private:
    const PNode*            bindNode;           //!< a node over which to display this 2D-element
    Vector2D                size;               //!< The size of the rendered item
    Vector2D*               toSize;             //!< The Size to iterate to.

    E2D_ALIGNMENT           alignment;          //!< How the Element is aligned around its Position

    bool                    bVisible;           //!< If the given Element2D is visible.
    bool                    bCurrentlyVisible;  //!< Evaluated in the TICK process, to see if the Element is Currently visible.
    bool                    bActive;            //!< If the given Element2D is active.
    E2D_LAYER               layer;              //!< What layer this Element2D is on.

    bool                    bRelCoorChanged;    //!< If Relative Coordinate has changed since last time we checked
    bool                    bRelDirChanged;     //!< If Relative Direction has changed since last time we checked

    Vector2D                relCoordinate;      //!< coordinates relative to the parent
    Vector2D                absCoordinate;      //!< absolute coordinates in the world ( from (0,0,0) )
    float                   relDirection;       //!< direction relative to the parent
    float                   absDirection;       //!< absolute diretion in the world ( from (0,0,1) )

    Vector2D                prevRelCoordinate;  //!< The last Relative Coordinate from the last update-Cycle.
    Vector2D                lastAbsCoordinate;  //!< this is used for speedcalculation, it stores the last coordinate
    float                   prevRelDirection;   //!< The last Relative Direciton from the last update-Cycle.

    Vector2D                velocity;           //!< Saves the velocity.

    Vector2D*               toCoordinate;       //!< a position to which to iterate. (This is used in conjunction with setParentSoft.and set*CoorSoft)
    float*                  toDirection;        //!< a direction to which to iterate. (This is used in conjunction with setParentSoft and set*DirSoft)
    float                   bias;               //!< how fast to iterate to the given position (default is 1)

    Element2D*              parent;             //!< a pointer to the parent node
    std::list<Element2D*>   children;           //!< list of the children of this Element2D

    unsigned int            parentMode;         //!< the mode of the binding

    static Element2D*       nullElement;        //!< The top-most Element
};

#endif /* _ELEMENT_2D_H */
