/*!
 * @file resource_manager.h
  *  The Resource Manager checks if a file/resource is loaded.

    If a file/resource was already loaded the resourceManager will
    return a void pointer to the desired resource.
    Otherwise it will instruct the coresponding resource-loader to load,
    and receive a pointer to it.

    it is possible to compile the resource Manager without some modules by
    just adding the compile flag -D....
    (NO_MODEL)
    (NO_AUDIO)
    (NO_TEXT)
    (NO_TEXTURES)
    (NO_SHADERS)
*/

#ifndef _RESOURCE_MANAGER_H
#define _RESOURCE_MANAGER_H

#include "base_object.h"
#include "filesys/file.h"

#include "multi_type.h"
#include <vector>

//! An eumerator for different fileTypes the resourceManager supports
typedef enum ResourceType
{
#ifndef NO_MODEL
  OBJ,                  //!< loading .obj file
  PRIM,                 //!< loading primitive model
  MD2,                  //!< loading md2-file
#endif /* NO_MODEL */
#ifndef NO_TEXT
  TTF,                  //!< loading a TrueTypeFont
#endif /* NO_TEXT */
#ifndef NO_AUDIO
  WAV,                  //!< loading wav
  MP3,                  //!< loading mp3
  OGG,                  //!< loading ogg
#endif /* NO_AUDIO */
#ifndef NO_TEXTURES
  IMAGE,                //!< loading an image
#endif /* NO_TEXTURES */
#ifndef NO_SHADERS
  SHADER,               //!< openGL-shader program
#endif /* NO_SHADERS */
  RESOURCE_TYPE_SIZE
};

//! An enumerator for different UNLOAD-types.
/**
   RP_NO:        will be unloaded on request
   RP_LEVEL:     will be unloaded at the end of a Level
   RP_CAMPAIGN:  will be unloaded at the end of a Campaign
   RP_GAME:      will be unloaded at the end of the whole Game (when closing orxonox)
*/
typedef enum ResourcePriority
{
  RP_NO        =   0,
  RP_LEVEL     =   1,
  RP_CAMPAIGN  =   2,
  RP_GAME      =   4
};

//! A Struct that keeps track about a resource its name its Type, and so on
struct Resource
{
  BaseObject*       pointer;           //!< Pointer to the Resource.
  unsigned int      count;             //!< How many times this Resource has been loaded.

  std::string       name;              //!< Name of the Resource.
  ResourceType      type;              //!< ResourceType of this Resource.
  ResourcePriority  prio;              //!< The Priority of this resource. (This will only be increased)

  MultiType         param[3];          //!< The Parameters given to this Resource.
};


//! The ResourceManager is a class, that decides if a file/resource should be loaded
/**
 * If a file/resource was already loaded the resourceManager will
 * return a pointer to the desired resource.
 * Otherwise it will instruct the corresponding resource-loader to load,
 * and receive the pointer to it.
 *
 * It does it by looking, if a desired file has already been loaded.
 * There is also the possibility to check for some variables
 */
class ResourceManager : public BaseObject
{
 public:
  virtual ~ResourceManager();
  /** @returns a Pointer to the only object of this Class */
  inline static ResourceManager* getInstance() { if (!singletonRef) singletonRef = new ResourceManager();  return singletonRef; };

  bool setDataDir(const std::string& dataDir);
  /** @returns the Name of the data directory */
  inline const std::string& getDataDir() const { return this->dataDir; };


  bool tryDataDir(const std::string& dataDir);
  bool verifyDataDir(const std::string& fileInside);
  bool addImageDir(const std::string& imageDir);

  bool cache(const std::string& fileName, ResourceType type, ResourcePriority prio = RP_NO,
             const MultiType& param0 = MultiType(), const MultiType& param1 = MultiType(), const MultiType& param2 = MultiType());
  BaseObject* copy(BaseObject* resourcePointer);

  BaseObject* load(const std::string& fileName, ResourcePriority prio = RP_NO,
                   const MultiType& param0 = MultiType(), const MultiType& param1 = MultiType(), const MultiType& param2 = MultiType());
  BaseObject* load(const std::string& fileName, ResourceType type, ResourcePriority prio = RP_NO,
                   const MultiType& param0 = MultiType(), const MultiType& param1 = MultiType(), const MultiType& param2 = MultiType());
  bool unload(BaseObject* pointer, ResourcePriority prio = RP_NO);
  bool unload(Resource* resource, ResourcePriority = RP_NO);
  bool unloadAllByPriority(ResourcePriority prio);

  Resource* locateResourceByInfo(const std::string& fileName, ResourceType type,
                                 const MultiType& param0 = MultiType(), const MultiType& param1 = MultiType(), const MultiType& param2 = MultiType()) const;
  Resource* locateResourceByPointer(const void* pointer) const;

  std::string toResourcableString(unsigned int i);
  bool fromResourceableString(const std::string& resourceableString);
  /** @returns the Count of Resources the ResourceManager handles */
  unsigned int resourceCount() const { return this->resourceList.size(); }

  void debug() const;


  // utility functions for handling files in and around the data-directory
  static std::string getFullName(const std::string& fileName);
  static bool isInDataDir(const std::string& fileName);

  static const char* ResourceTypeToChar(ResourceType type);
  static ResourceType stringToResourceType(const std::string& resourceType);

 private:
  ResourceManager();
  Resource* loadResource(const std::string& fileName, ResourceType type, ResourcePriority prio,
                         const MultiType& param0, const MultiType& param1, const MultiType& param2);

 private:
  static ResourceManager*    singletonRef;       //!< singleton Reference

  std::string                dataDir;            //!< The Data Directory, where all relevant Data is stored.
  std::vector<std::string>   imageDirs;          //!< A list of directories in which images are stored.

  std::vector<Resource*>     resourceList;       //!< The List of Resources, that has already been loaded.

  static const char*         resourceNames[RESOURCE_TYPE_SIZE];
};

#endif /* _RESOURCE_MANAGER_H */
