/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */

#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#define PARSELINELENGHT     512       //!< how many chars to read at once

#include "filesys/file.h"
#include <list>

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser : public File
{
  private:
    ////////////////////////////////////
    //! a struct for Entries in the Parser's File's Sections
    struct IniEntry
    {
      std::string         comment;  //!< A Comment that is appendet to the Top of this Entry.
      std::string         name;     //!< name of a given Entry
      std::string         value;    //!< value of a given Entry
    };

    //! a struct for Sections in the Parser's file
    struct IniSection
    {
      std::string         comment;  //!< A Comment that is appendet to the Top of this Section.
      std::string         name;     //!< name of a given section
      std::list<IniEntry> entries;  //!< a list of entries for this section
    };
    ////////////////////////////////////

  public:
    IniParser (const std::string& filename = "");
    virtual ~IniParser ();

    /** @returns true if the file is opened, false otherwise*/
    bool isOpen() const { return true; } ///HACK //(this->fileName.empty()) ? false : true; };
    /** @returns the fileName we have opened. */
    const std::string& getFileName() const { return this->fileName; };

    bool readFile(const std::string& fileName);
    bool writeFile(const std::string& fileName) const;

    void setFileComment(const std::string& fileComment);
    const std::string& getFileComment() const { return this->comment; };

    bool addSection(const std::string& sectionName);
    bool getSection(const std::string& sectionName);
    void setSectionComment(const std::string& comment, const std::string& sectionName);
    const std::string& getSectionComment(const std::string& sectionNane) const;

    // iterate through sections with these Functions
    void firstSection();
    const std::string& nextSection();


    bool addVar(const std::string& entryName, const std::string& value, const std::string& sectionName = "" );
    const std::string& getVar(const std::string& entryName, const std::string& sectionName, const std::string& defaultValue = "") const;
    bool IniParser::editVar(const std::string& entryName, const std::string& value, const std::string& sectionName = "", bool createMissing = true);
    void setEntryComment(const std::string& comment, const std::string& entryName, const std::string& sectionName);
    const std::string& getEntryComment(const std::string& entryName, const std::string& sectionName) const;

    // iterate Through Variables with these Functions.
    void firstVar();
    bool nextVar();


    // retrieving functions when iterating.
    const std::string& getCurrentSection() const;
    const std::string& getCurrentName() const;
    const std::string& getCurrentValue() const;


    // maintenance.
    void debug() const;


  private:
    void deleteSections();
    void setFileName(const std::string& fileName);

    void setFileComment();
    void setSectionComment();
    void setEntryComment();

    std::list<IniSection>::const_iterator getSectionIT(const std::string& sectionName) const;
    std::list<IniSection>::iterator getSectionIT(const std::string& sectionName);

    std::list<IniEntry>::const_iterator getEntryIT(const std::string& entryName, const std::string& sectionName = "") const;
    std::list<IniEntry>::iterator getEntryIT(const std::string& entryName, const std::string& sectionName = "");

  private:
    std::string                      fileName;        //!< The name of the File that was parsed.
    std::string                      comment;         //!< A Comment for the header of this File.
    std::list<IniSection>            sections;        //!< a list of all stored Sections of the Parser
    std::list<IniSection>::iterator  currentSection;  //!< the current selected Section
    std::list<IniEntry>::iterator    currentEntry;    //!< the current selected entry (in currentSection)

    std::list<std::string>           commentList;     //!< A list of Comments. (this is for temporary saving of Comments, that are inserted in front of Sections/Entries.)


    static const std::string         emptyString;
};

#endif /* _INI_PARSER_H */
