/*
	orxonox - the future of 3D-vertical-scrollers
 
	Copyright (C) 2006 orx
 
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.
 
	### File Specific:
	main-programmer: Marco Biasini
 
 */
#include "terrain_page.h"
#include "terrain.h"
#include "glincl.h"
#include <stdio.h>
#include <math.h>

#define CHECK_GL_ERROR( _d ) do { \
	GLenum __err = glGetError(); \
	if ( __err != GL_NO_ERROR ) \
		printf( "check%s: %s\n", _d, (char*)gluErrorString( __err ) );\
	}\
	while ( 0 )

TerrainPage::TerrainPage( Terrain *_owner, int _xOffset, int _zOffset )
: TerrainQuad( _owner, _xOffset, _zOffset )
{
	scale = owner->getScale();
	numVertices = numIndices = 0;
	errors = new LODError[TerrainPage::MAX_LODS];
	vertices = NULL; indices = NULL; coords = NULL;
	position = Triple( scale.x*_xOffset, 0.0f, scale.z*_zOffset );
	isVisible = false;
	next = NULL;
	active = false;
	previous = NULL;
	currentLOD = -1;
	forceTesselation = true;
}

int TerrainPage::chooseLOD()
{
	wantedLOD = -1;
	Triple cam( owner->getCameraPosition() );
	
	for ( int i = TerrainPage::MAX_LODS-1; i >= 0; --i ) {
		Triple distance( cam.x-errors[i].correct.x,
						 cam.y-errors[i].correct.y,
						 cam.z-errors[i].correct.z );
		
		float d = distance.length();
		
		assert( d > 0.000001f );
		float err =  errors[i].diff / d ;
		
		if ( err*scale.y < owner->getDetail() ) {
			wantedLOD = i;
			break;
		}	
	}
	if ( wantedLOD < 0 ) {
		wantedLOD = TerrainPage::MAX_LODS-1;
	}	
	// Calculate the tween factor. This is differently if the LOD is 0.
	if ( wantedLOD > 0 ) {
		
	}
	else {
		
	}
	return wantedLOD;
}

void TerrainPage::activate()
{
	
	pTerrainPage list = owner->getActiveList();
	next = list;
	if ( list )
		list->previous = this;
	owner->incActivated();
	owner->setActiveList( this );
}

void TerrainPage::deactivate()
{
	pTerrainPage list = owner->getActiveList();
	
	if ( previous ) {
		previous->next = next;
	}
	if ( next ) {
		next->previous = previous;
	}
	if ( list == this )
		owner->setActiveList( next );
	next = NULL;
	owner->incDeactivated();
	previous = NULL;
	
}

void TerrainPage::calculateError( int _lod )
{
	float	sumError = 0.0f;
	int		numErrors = 0;
	int size = owner->getPageSize();
	if( _lod!=0 )
	{
		int pow = 1 << _lod, x0, y0, xi, yi;
		// Altough these four nested loops look very scary, they're not 
		// that bad. 
		for( y0 = 0 ; y0 < size-pow; y0 += pow ) {
			for( x0 = 0; x0 < size-pow; x0 += pow ) {
				for( yi = 1; yi < pow; yi++ ) {
					for( xi = 1; xi < pow; xi++ )
					{
						int	x = x0+xi,
						y = y0+yi;
						float	fx0 = ( float )xi/( float )pow, fx1 = 1.0f-fx0,
								fy0 = ( float )yi/( float )pow, fy1 = 1.0f-fy0;
						
						float	height00 = getAltitude( x0, y0 ),
								height10 = getAltitude( x0+pow,y0 ),
								height01 = getAltitude( x0,y0+pow ),
								height11 = getAltitude( x0+pow, y0+pow );
						
						float	paintHeight =	fx1*fy1 * height00 +
							fx0*fy1 * height10 +
							fx1*fy0 * height01 +
							fx0*fy0 * height11,
							correctHeight =  getAltitude( x, y );
						
						float	er = ( float )fabs( correctHeight - paintHeight );
						
						numErrors++;
						sumError += er;
					}
				}
			}
		}
		float error = sumError / numErrors;
		getVertex(size/2, size/2, errors[_lod].correct);
		errors[_lod].real = errors[_lod].correct;
		errors[_lod].real.y += error;
		errors[_lod].diff = error;
	}
}

void TerrainPage::calculateBounds()
{
	int size = owner->getPageSize();
	float alt = 0.0f;
	
	Triple	min( xOffset*scale.x ,0.0f, zOffset*scale.z ), 
			max( (xOffset+1)*scale.x, 0.0f, (zOffset+1)*scale.z);
	
	min.y = max.y = getAltitude( 0, 0 );
	for ( int x = 0; x < size; ++x ) {
		for ( int z = 0; z < size; ++z ) {
			alt = getAltitude( x, z );
			min.y = fmin( min.y, alt );
			max.y = fmax( max.y, alt );
		}
	}
	bounds.set( min, max );
}

float TerrainPage::getAltitude( int _x, int _z ) const
{
	assert( _x >= 0 && _x < 17 );
	assert( _z >= 0 && _z < 17 );
	return position.y+scale.y*owner->getAltitude( 
			_x+(owner->getPageSize()-1)*xOffset, 
			_z+(owner->getPageSize()-1)*zOffset );
}

void TerrainPage::updateTesselation( )
{
	currentLOD = wantedLOD;
	if ( forceTesselation || currentLOD != currentLOD ) {
		tesselate( currentLOD );
	}	

	forceTesselation = false;
}

/**
 * Genererates a tesselation for the given _lod. 
 */
void TerrainPage::tesselate( int _lod )
{
	int count = owner->getPageSize()*owner->getPageSize();	
	memset( indexHash, 0xffff, 
			sizeof(unsigned short)*count );
	numVertices = 0; numIndices = 0;
	
	assert( isVisible );
	//Calculate the pace, based on the lod.
	int stride = 1 << _lod;
	for ( int z = 0; z < owner->getPageSize()-stride; z+=stride ) {
		if ( z > 0 ) {			
			//Connect the two rows together by inserting the last index of last row
			//and the first index of the next row two times. This will produce two
			//degenerate triangles.
			addAgain( );
			addIndex( getIndex( 0, z ) );
		}
		for ( int x = 0; x < owner->getPageSize(); x+=stride ) {
			addIndex( getIndex( x, z ) );
			addIndex( getIndex( x, z+stride ) );
		}
	}
}


void TerrainPage::show( )
{
	int count = owner->getPageSize()*owner->getPageSize();
	
	vertices = new Triple[count];
	// Not the economical way, but we just want to have space for all indices.
	indices = new unsigned short[count*3];
	indexHash = new unsigned short[count];	
	coords = new TexCoord[count];
	forceTesselation = true;
	activate();
	active = true;
}

void TerrainPage::hide( )
{
	if ( vertices ) {
		delete[] vertices;
		vertices = NULL;
	}
	delete[] coords;	
	if ( indices ) {
		delete[] indices;
		indices = NULL;
		numIndices = 0;
	}
	deactivate();
}

void TerrainPage::drawBox()
{
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();
	glTranslatef( bounds.corner.x, bounds.corner.y, bounds.corner.z );
	glBegin( GL_QUADS );
		glVertex3f( 0.0f, 0.0f, 0.0f );
		glVertex3f(  bounds.x, 0.0f, 0.0f );
		glVertex3f( bounds.x, bounds.y, 0.0f );
		glVertex3f( 0.0f, bounds.y, 0.0f );
		
		glVertex3f( 0.0f, 0.0f, bounds.z );
		glVertex3f(  bounds.x, 0.0f, bounds.z );
		glVertex3f( bounds.x, bounds.y, bounds.z );
		glVertex3f( 0.0f, bounds.y, bounds.z );		
		
		glVertex3f( 0.0f, 0.0f, 0.0 );
		glVertex3f(  0.0, 0.0f, bounds.z );
		glVertex3f( 0.0f, bounds.y, bounds.z );
		glVertex3f( 0.0f, bounds.y, 0.0f );				
		
		glVertex3f( bounds.x, 0.0f, 0.0 );
		glVertex3f(  bounds.x, 0.0f, bounds.z );
		glVertex3f( bounds.x, bounds.y, bounds.z );
		glVertex3f( bounds.x, bounds.y, 0.0f );						
	glEnd();
	glPopMatrix();
}

void TerrainPage::draw( )
{
	//These give an error
	assert( glIsEnabled( GL_VERTEX_ARRAY ) );
	assert( !glIsEnabled( GL_NORMAL_ARRAY ) );
	assert( isVisible ); assert( numIndices > 0 );
	assert( active );
	active = false;
	CHECK_GL_ERROR( "1" );	
	glVertexPointer( 3, GL_FLOAT, 0, vertices );
	glTexCoordPointer( 2, GL_FLOAT, 0, coords );	
	glDrawElements( GL_TRIANGLE_STRIP, numIndices, 
					GL_UNSIGNED_SHORT, indices );
	
	if ( owner->debug() )
		drawBox( );
		
	CHECK_GL_ERROR( "2" );	
}

short TerrainPage::getIndex( int _x, int _z )
{
	unsigned short index = _z*owner->getPageSize()+_x;
	if ( indexHash[index] == 0xffff ) {
		//The vertex didn't exists before, lets create it.
		indexHash[index] = numVertices;
		getVertex( _x, _z, vertices[numVertices] );
		getCoord( _x, _z, coords[numVertices] );
		numVertices++;
	}
	return indexHash[index];
}

void TerrainPage::getCoord( int _x, int _z, TexCoord& _coord ) const 
{
	owner->getCoord( _x+xOffset*(owner->getPageSize()-1 ), 
					 _z+zOffset*(owner->getPageSize()-1 ), _coord );
}

void TerrainPage::getVertex( int _x, int _z, Triple& _vertex ) const
{
	_vertex.x = position.x+scale.x*_x/
		( owner->getPageSize()-1 );
	_vertex.y = getAltitude( _x, _z );
	_vertex.z = position.z+scale.z*_z/( owner->getPageSize()-1 );
}

void TerrainPage::calculateErrors()
{
	for ( int i = 0; i < TerrainPage::MAX_LODS; ++i )
		calculateError( i );
}