
#include "terrain.h"
#include "terrain_page.h"
#include "glincl.h"
#include "util/loading/resource_manager.h"
#include "debug.h"

#ifdef HAVE_SDL_SDL_IMAGE_H
#include <SDL/SDL_image.h>
#else
#include <SDL_image.h>
#endif

void Terrain::build()
{
	ResourceManager *MANAGER = ResourceManager::getInstance();
	std::string full = MANAGER->getFullName( heightmapSource );
	SDL_Surface *tmpData = IMG_Load( full.c_str() );
 	if ( !tmpData ) {
		PRINTF(0)( "I' sorry, I can't load %s\n", full.c_str() );
		return;
	}		
	
	heightfield.height = tmpData->h;
	heightfield.width = tmpData->w;
	heightfield.pitch = tmpData->pitch;
	int dataSize = heightfield.pitch*heightfield.height;
	heightfield.data = new UByte[dataSize];
	memcpy( heightfield.data, tmpData->pixels, sizeof(UByte)*dataSize );
	SDL_FreeSurface( tmpData );
	//TODO: Perform some checks...
	pagesX = (heightfield.width/(pageSize-1) );
	pagesZ = (heightfield.height/(pageSize-1) );
	tex = (Texture*)MANAGER->load( lightmapSource );
	
	//TODO: Determine layer visibility!
	printf( "creating terrain pages ( %d, %d )...", pagesX, pagesZ );
	pages = new pTerrainPage[pagesX*pagesZ];	
	for ( int x = 0; x < pagesX; ++x )
		for ( int z = 0; z < pagesZ; ++z )
			pages[z*pagesX+x] = createPage( x, z );
	printf( "looks good\n" );
	printf( "creating quad_tree data structure..." );
	root = createQuadTree( 0, 0, pagesX, pagesZ );
	activePages = NULL;
	printf( "looks good\n" );
}

pTerrainQuad Terrain::createQuadTree( int _x0, int _z0, int _x1, int _z1, int _depth )
{
	int _x01 = (_x1+_x0)/2; int _z01 = (_z1+_z0)/2;
	pTerrainQuad node;
	if ( _x1-_x0 == 1 ) {
		node = getPage( _x0, _z0 );
		node->setChildren( NULL, NULL, NULL, NULL );
	}
	else {
		assert( ( _x0 % 2 ) == 0 ); 
		assert( ( _x1 % 2 ) == 0 ); 
		assert( ( _z0 % 2 ) == 0 ); 
		assert( ( _z1 % 2 ) == 0 );		
		node = new TerrainQuad( this, _x0, _z0, _x1, _z1 );
		node->setChildren( 
			createQuadTree( _x0, _z0, _x01, _z01, _depth+1 ),
			createQuadTree( _x01, _z0, _x1, _z01, _depth+1 ), 
			createQuadTree( _x0, _z01, _x01, _z1, _depth+1 ), 
			createQuadTree( _x01, _z01, _x1, _z1, _depth+1 ) );
	}
	node->calculateBounds();
	return node;
}

pTerrainPage Terrain::createPage( int _xOffset, int _zOffset ) const
{
	pTerrainPage newPage = new TerrainPage( const_cast<Terrain*>( this ), _xOffset, _zOffset );
	newPage->setScale( scale );	
	newPage->setPosition( Triple( scale.x*_xOffset, 0.0f, scale.z*_zOffset ) );
	newPage->calculateErrors();
	return newPage;
}

//TODO: The method could be optimized. We could get rid of the hidePages call
//		by setting a active flag in every pages that is visible. Then a walk
//		through the activePages list is sufficient to remove all invisible 
//		pages.
//		DONE
void Terrain::determineVisiblePages( pTerrainQuad _node )
{
	switch( _node->cull() ) {
		case Frustum::INTERSECT:
			//printf( "partially inside frustum\n" );
			if ( !_node->isChildless() ) {
				pTerrainQuad *children = _node->getChildren();
				for ( int i = 0; i < 4; ++i, ++children )
					determineVisiblePages( *children );
			}
			else {
				showPages( _node->getXOffset(), 
						   _node->getZOffset(), 1, 1 );
			}
			break;
		case Frustum::INSIDE:
			//printf( "fully inside frustum\n" );			
			showPages( _node->getXOffset(), 
					   _node->getZOffset(), 
					   _node->getWidth() , 
					   _node->getHeight() );
			break;
		case Frustum::OUTSIDE:
			cullCount+= (_node->getWidth()-1)*(_node->getHeight()-1);
			break;
	}
}

void Terrain::draw( )
{
	glGetError();
	pTerrainPage page = NULL;
	frustum->extractPlanes();
	glEnableClientState( GL_VERTEX_ARRAY );

	glEnableClientState( GL_TEXTURE_COORD_ARRAY ); 
	glEnable( GL_TEXTURE_2D );
	glEnable( GL_DEPTH_TEST );
	glDisable( GL_CULL_FACE );
	glBindTexture( GL_TEXTURE_2D, tex->getTexture() );
	glDisable( GL_LIGHTING );
	glEnable( GL_COLOR_MATERIAL );
	glColor3f( 1.0f, 1.0f, 1.0f );
	cullCount = 0;
	// Lets see which pages are visible.
	determineVisiblePages( root );
		
	int count = 0;
	page = activePages;	
	while ( page ) {
		if ( !page->isActive() ) {
			pTerrainPage tmp = page;
			page = tmp->getNext();
			tmp->setVisibility( false );
			tmp->deactivate();
			deactivatedCount++;
			continue;
		}	
		count++;
		page->updateTesselation();
		//Draw the page. TODO: It would be nice if all the pages with an LOD of 4 could be merged
		//in one index buffer. This would give us a speed boost for sure...
		page->draw();
		page = page->getNext();
	}
	//printf( "%d pages activated, %d pages deactivated\n", activatedCount, deactivatedCount );
	activatedCount = 0; deactivatedCount = 0;
	float percentage = (float)cullCount/(float)(pagesX*pagesZ)*100.0f;
	//printf( "culled %f%% terrain pages away\n",  percentage );
}



void Terrain::getAltitude( Triple& _alt, Triple& _normal )
{
	float xScaled = _alt.x / scale.x, zScaled = _alt.z / scale.z;
	//The offset on the map
	int xOff =  (int)xScaled, zOff = (int)zScaled;
	//The interpolation values.
	float u = xScaled-xOff, v = zScaled-zOff;
	
	float dX = scale.x / ( pageSize-1 );
	float dZ = scale.z / ( pageSize-1 );
	
	//If u is bigger than v, we are on the lower triangle...
	if ( u > v ) {
		
		float alt[] = {
			getAltitude( xOff+0, zOff+0 )*scale.y,
			getAltitude( xOff+1, zOff+0 )*scale.y,
			getAltitude( xOff+1, zOff+1 )*scale.y };
		_alt.y = (1.0f-u)*(1.0-v)*alt[0]+u*(1.0f-v)*alt[1]+u*v*alt[2];
		
		//Since we know about the directions of some x and z-coordinates, 
		//not the whole cross products needs to be calculated. Grab yourself
		//pen and paper :)
		_normal.x = -dZ*( alt[0] - alt[1] );				
		_normal.y = -dZ*dX;
		_normal.z =  dX*( alt[2] - alt[1] );
	}
	else {
		float alt[] = {
			getAltitude( xOff+0, zOff+0 )*scale.y,
			getAltitude( xOff+0, zOff+1 )*scale.y,
			getAltitude( xOff+1, zOff+1 )*scale.y };			
		_alt.y = (1.0f-u)*(1.0-v)*alt[0]+(1.0f-u)*v*alt[1]+u*v*alt[2];
		//Since we know about the directions of some x and z-coordinates, 
		//not the whole cross products needs to be calculated. Grab yourself
		//pen and paper :)
		_normal.x =  dZ*( alt[2] - alt[1] );				
		_normal.y = -dZ*dX;
		_normal.z = -dX*( alt[0] - alt[1] );
	}
}

void Terrain::showPages( int _x0, int _z0, int _width, int _height )
{
	for ( int x = 0; x < _width; ++x ) {
		for ( int z = 0; z < _height; ++z ) {
			pTerrainPage page = getPage( _x0+x, _z0+z );
			page->setVisibility( true );			
			page->chooseLOD();

		}
	}
}
