/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "texture_sequence.h"

#include "debug.h"
#include "graphics_engine.h"

#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif

/**
 *  Constructor for a Texture
 */
TextureSequence::TextureSequence(unsigned int count, ...)
{
  this->setClassID(CL_TEXTURE_SEQUENCE, "TextureSequence");

  va_list textureNameList;
  va_start(textureNameList, count);

  this->loadImageSeries(count, textureNameList);
}

/**
 * Destructor of a TextureSequence
 *
 * Frees Data, and deletes the textures from GL
 */
TextureSequence::~TextureSequence()
{
  this->clearLists();
}

void TextureSequence::clearLists()
{
  // delete all images
  while(!this->images.empty())
  {
    SDL_FreeSurface(this->images.back());
    this->images.pop_back();
  }

  // delete all textures.
  while(!this->textures.empty())
  {
    if (glIsTexture(this->textures.back()))
      glDeleteTextures(1, &this->textures.back());
    this->textures.pop_back();
  }
}

/**
 * @brief rebuilds all the textures from the Images stored in this FrameSequence
 */
bool TextureSequence::rebuild()
{
  PRINTF(3)("Reloading TextureSequence of %s '%s'\n", this->getClassName(), this->getName());

  for (unsigned int i = 0; i < this->textures.size(); i++)
  {
    if (glIsTexture(this->textures[i]))
    {
      glDeleteTextures(1, &this->textures[i]);
      this->textures[i] = 0;
    }

    if (this->images[i] != NULL)
      this->textures[i] = loadTexToGL(this->images[i]);
  }
  return true;
}

/**
 * @brief loads an image Sequence
 * @param count how many images to load to the TextureSequence
 * @param ... the names of the Images to load
 * @returns true on success, false otherwise
 */
bool TextureSequence::loadImageSeries(unsigned int count, ...)
{
  va_list textureNameList;
  va_start(textureNameList, count);

  return this->loadImageSeries(count, textureNameList);
}

/**
 * @brief loads an image Sequence
 * @param count how many images to load to the TextureSequence
 * @param textures the names of the Images to load
 * @returns true on success, false otherwise
 */
bool TextureSequence::loadImageSeries(unsigned int count, va_list textures)
{
  bool retVal = true;
  for (unsigned int i = 0; i < count; i++)
  {
    if( !this->addFrame(va_arg(textures, char*)))
      retVal = false;
  }
  return retVal;
}

bool TextureSequence::loadImageSeries(const std::string& imagePrefix, unsigned int from, unsigned int to, const std::string& extension)
{
  unsigned int index = 0;
  unsigned int frameSize = 0;
  // search for the special character # in the LoadParam
  if ((index = imagePrefix.find("_#")) != std::string::npos)
  {
    std::string _imagePrefix = imagePrefix;
    index++; // start at #
    while(imagePrefix[index+frameSize] == '#')
    {
      _imagePrefix[index+frameSize] = '0';
      frameSize++;
    }

    PRINTF(4)("Found %d '#'s in %s... searching for LOD's\n", frameSize, imagePrefix.c_str());
    char tmpString[32];
    for (unsigned int i = from; i < to; i++)
    {
      sprintf(tmpString, "%d", i);
      _imagePrefix.replace(index+frameSize -strlen(tmpString), index+frameSize, tmpString);
      printf("TEST STRING %s\n", _imagePrefix.c_str());
    }
    return true;
  }
  return false;
}

/**
 * @brief adds a new Frame to this Sequence (at the end)
 * @param imageName the Name of the Image to add
 * @returns true on success
 */
bool TextureSequence::addFrame(const std::string& imageName)
{
  SDL_Surface* addSurface = IMG_Load(imageName.c_str());
  bool success = this->addFrame(addSurface);
  delete addSurface;

  return success;
}

/**
 * @brief adds a new Frame at the end of the Sequence.
 * @param surface the Surface to add at the end of the Sequence.
 */
bool TextureSequence::addFrame(SDL_Surface* surface)
{
  if (surface == NULL)
    return false;
  bool hasAlpha;
  SDL_Surface* newSurf = this->prepareSurface(surface, hasAlpha);
  if (newSurf != NULL)
  {
    this->images.push_back(newSurf);
    this->textures.push_back(Texture::loadTexToGL(newSurf));
  }
  this->setAlpha(hasAlpha);

  return true;
}

/**
 * @brief adds a new Frame at the end of the Sequence.
 * @param texture the texture to add at the end of the Sequence.
 */
bool TextureSequence::addFrame(GLuint texture)
{
  if (texture == 0)
    return false;
  this->textures.push_back(texture);

  return true;
}

/**
 * @brief moves to the n'th texture which can then be retrieved via the Texture function: this->getTexture()
 * @param frameNumber the n-th frame
 */
/*void TextureSequence::gotoFrame(unsigned int frameNumber)
{
  if (this->textures.size() > frameNumber)
    this->setTexture(this->textures[frameNumber]);
}
*/
