/*!
 * @file threading.h
 * @brief Definition of Thread Classes.
 *
 * These are mainly Classes, that are used for wrapping around SDL_thread
*/

#ifndef _THREADING_H
#define _THREADING_H

#ifdef HAVE_SDL_H
 #include <SDL_thread.h>
#else
 #include <SDL/SDL_thread.h>
#endif
#include <string>
namespace OrxThread
{
  //! A class for Wrapping Threads
  class Thread
  {

  public:
    Thread(int (*fn)(void *), void *data) { this->thread = SDL_CreateThread(fn, data); };
    virtual ~Thread() { SDL_KillThread(this->thread); }
    void exit ( int returnCode = 0 );
    bool isFinished () const;
    bool isRunning () const;
    void wait() { SDL_WaitThread(this->thread, NULL); };

    void start();
    void terminate();
  private:
    SDL_Thread* thread;

  };

  class Mutex
  {
  public:
    Mutex() {  this->mutex = SDL_CreateMutex(); };
    ~Mutex() { SDL_DestroyMutex(this->mutex); }

    void lock() { SDL_mutexP(mutex); };
    void unlock() { SDL_mutexV(mutex); };

    SDL_mutex* getMutex() const { return this->mutex; };
  private:
    SDL_mutex* mutex;
  };

  //! A Class that locks a Mutex within its scope
  class MutexLock
  {
  public:
    //! Locks the Mutex mutex in this Scope.
    MutexLock(Mutex* mutex) { SDL_mutexP(mutex->getMutex()); this->mutex = mutex; };
    ~MutexLock() { SDL_mutexV(mutex->getMutex()); };
  private:
    Mutex* mutex;         //!< The Mutex to lock.
  };
}

#endif /* _THREADING_H */
