/*!
 * @file state.h
 * Definition of the States Class
*/

#ifndef _STATE_H
#define _STATE_H


// FORWARD DECLARATION
class PNode;
class Camera;
class CameraTarget;
class WorldEntity;
class Player;
class SkyBox;
class StoryEntity;
class ObjectManager;
class GameRules;

class ScriptManager;


//! handles states about orxonox's most importatn objects
/**
 * This is an abstract Class-container, not really a Class.
 * in this Class only static references to the most important
 * Objects/List/etc. are stored.
 */
class State {

 public:
   //////////////
   /// CAMERA ///
   //////////////
  /** @param camera the PNode to the Camera, @param cameraTarget the PNode to the Camera's target */
  static void setCamera(Camera* camera, CameraTarget* cameraTarget);
  static inline Camera* getCamera() { return State::camera; };
  static inline CameraTarget* getCameraTarget() { return State::cameraTarget; };
  /** @returns a Pointer to the PNode of the Camera */
  static inline PNode* getCameraNode() { return State::cameraNode; };
  /** @returns a Pointer to the CameraTarget */
  static inline PNode* getCameraTargetNode() { return State::cameraTargetNode; };

  ////////////////
  /// SKYBOX   ///
  ////////////////
  /** @returns the current SkyBox */
  static inline SkyBox* getSkyBox() { return State::skyBox; };
  /** @param skyBox the SkyBox */
  static inline void setSkyBox(SkyBox* skyBox) { State::skyBox = skyBox; };

  //////////////////////
  /// OBJECT-MANAGER ///
  //////////////////////
  /** @param objectManager the new Current ObjectManager */
  static inline void setObjectManager(ObjectManager* objectManager) { State::objectManager = objectManager; };
  /** @returns the current ObjectManager. */
  static inline ObjectManager* getObjectManager() { return State::objectManager; };

  static inline void setResolution(unsigned int resX, unsigned int resY) { State::resX = resX; State::resY = resY; };
  static inline unsigned int getResX() { return State::resX; };
  static inline unsigned int getResY() { return State::resY; };

  //////////////////////
  /// STORY-ENTITY   ///
  //////////////////////
  /** @param storyEntity sets the current StoryEntity that is been played */
  static inline void setCurrentStoryEntity(StoryEntity* storyEntity) { State::storyEntity = storyEntity; };
  /** @returns the current StoryEntity played */
  static inline StoryEntity* getCurrentStoryEntity() { return State::storyEntity; };

  /** @param gameRules sets the current GameRules */
  static inline void setGameRules(GameRules* gameRules) { State::gameRules = gameRules; }
  /** @returns the GameRules reference*/
  static inline GameRules* getGameRules() { return State::gameRules; }

  //////////////
  /// PLAYER ///
  //////////////
  /** @param player sets the current local player */
  static inline void setPlayer(Player* player) { State::player = player; };
  /** @returns the local player*/
  static inline Player* getPlayer() { return State::player; };


  ///////////////
  /// NETWORK ///
  ///////////////
  /** sets the online stat (multiplayer network) @param bOnline is true if node is online */
  static inline void setOnline(bool bOnline) { State::bOnline = bOnline; }
  /** @returns true if this node is online (multiplayer network game) */
  static bool isOnline() { return State::bOnline; }



  ////////////////////
  /// SCRIPT_ENGINE ///
  ////////////////////
  static void setScriptManager(ScriptManager* scriptManager) { State::scriptManager = scriptManager; };
  static ScriptManager* getScriptManager() { return State::scriptManager; };

  ////////////
  /// Menu ///
  ////////////
  /** sets the menu mode @param mode true if always exit to menu */
  static inline void setMenuID(int menuID) { State::menuID = menuID; }
  /** @returns the menu mode */
  static inline int getMenuID() { return State::menuID;}



 private:
  State();

  static Camera*                camera;             //!< The current Camera.
  static CameraTarget*          cameraTarget;       //!< The Camera Target.
  static PNode*                 cameraNode;         //!< A reference to the camera
  static PNode*                 cameraTargetNode;   //!< A reference to the cameraTarget
  static PNode*                 nullParent;         //!< A reference to the Null-PNode.
  static ObjectManager*         objectManager;      //!< A reference to the current ObjectManager
  static StoryEntity*           storyEntity;        //!< A reference to the current StoryEntity played
  static GameRules*             gameRules;          //!< A reference to the GameRules
  static Player*                player;             //!< A reference to the Player

  static SkyBox*                skyBox;            //!< The SkyBox used in the current world.

  static  ScriptManager*        scriptManager;     //!< The ScriptManager.

  static unsigned int           resX;              //!< The X Resolution of the screen.
  static unsigned int           resY;              //!< The Y Resolution of the screen.
  static int                    menuID;            //!< -1 on default, otherwise orxonox's Menu ID
  static bool                   bOnline;           //!< Is true if this node is in multiplayer mode (via network)
  };

#endif /* _STATE_H */
