#include <string>
#include <list>


#include "script_manager.h"
#include "lunar.h"

#include "class_list.h"

#include "script.h"
#include "script_trigger.h"
#include "luaincl.h"
#include "loading/load_param.h"
#include "parser/tinyxml/tinyxml.h"



ScriptManager::ScriptManager(const TiXmlElement* root)
{
  this->setName("ScriptManager");
  printf("ScriptManager created\n");
  this->scripts = NULL;
  this->triggers = NULL;

  if (root != NULL)
    this->loadParams(root);
}



ScriptManager::~ScriptManager()
{
  this->flush();
}


void ScriptManager::loadParams(const TiXmlElement* root)
{
  //BaseObject::loadParams(root);
  {
    LoadParamXML(root, "Scripts", this, ScriptManager, createScripts);

    LoadParamXML(root, "ScriptTriggers", this, ScriptManager, createTriggers);
  } // make shure that the loading process is finished

  // fill the scripts and triggers (doing that on runtime is very slow!)
  getTriggers();
  getScripts();
}



void  ScriptManager::flush()
{
  //Delete all scripts as they aren't deleted automatically
  if(this->getScripts())
    while(!scripts->empty())
      delete scripts->front();
}

void  ScriptManager::createScripts(const TiXmlElement* scripts)
{

  LOAD_PARAM_START_CYCLE(scripts, object);
  {
    new Script(object);
  }
  LOAD_PARAM_END_CYCLE(object);

}

void ScriptManager::createTriggers(const TiXmlElement* triggers)
{
  LOAD_PARAM_START_CYCLE(triggers, object);
  {
    new ScriptTrigger(object);
  }
  LOAD_PARAM_END_CYCLE(object);

}


Script* ScriptManager::getScriptByFile(const std::string& file)
{
  if(getScripts())
    for(std::list<BaseObject*>::const_iterator it = scripts->begin(); it != scripts->end(); it++ )
    {
      if( (dynamic_cast<Script*>(*it))->getFileName().compare(file) == 0)
      {
        return dynamic_cast<Script*>(*it);
      }
    }

  return NULL;

}


bool ScriptManager::getScripts()
{
  return (this->scripts != NULL || (this->scripts = ClassList::getList(CL_SCRIPT)) != NULL);
}

bool ScriptManager::getTriggers()
{
  return (this->triggers != NULL || (this->triggers = ClassList::getList(CL_SCRIPT_TRIGGER)) != NULL);
}



