/*!
 * @file executor.h
 * Definition of an Executor
 */

#ifndef _EXECUTOR_H
#define _EXECUTOR_H

#include "base_object.h"

#include "helper_functions.h"
#include "multi_type.h"
#include "substring.h"
#include "functor_list.h" //< MUST BE INCLUDED HERE AT LEAST ONCE.

//! an enumerator for the definition of the Type.
typedef enum {
  Executor_Objective         = 1,
  Executor_Static            = 2,

  Executor_NoLoadString      = 8,
} Executor_Type;

////////////////
// BASE CLASS //
////////////////
//! a BaseClass for all possible Executors
/**
 * An Executor is an Object, that is able to call Objects of Any type (class)
 * and execute a function with given parameters on it.
 *
 * The Executor is able to handle:
 *  Objects of any Class (Templated)
 *  Default Values
 *  Functions with up to 5 parameters (more seems useless)
 *  Functions with many types (@see functor_list.h)
 */
class Executor : public BaseObject
{
  public:
    virtual ~Executor();

    virtual Executor* clone () const = 0;
//    virtual bool operator==(const Executor* executor) const = 0;

    // SETTING up the EXECUTOR
    Executor* defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                            const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                            const MultiType& value4 = MT_NULL);
    /** @param i the i'th defaultValue, @returns reference to the MultiType */
    inline MultiType& getDefaultValue(unsigned int i) { return defaultValue[i]; };

    // EXECUTE
    /** executes a Command. @param objec the Object, @param count how many values, @param values the Values */
    virtual void operator()(BaseObject* object, int& count, void* values) const = 0;
    /** executes a Command @param object the object to apply this to @param parameters the parameters the command takes */
    virtual void operator()(BaseObject* object, const SubString& sub = SubString()) const = 0;

    // RETRIEVE INFORMATION
    /** @returns the Type of this Function (either static or objective) */
    inline long getType() const { return this->functorType; };
    /** @returns the Count of Parameters this Executor takes */
    inline unsigned int getParamCount() const { return this->paramCount; };

    static void debug();

  protected:
    Executor(const MultiType& param0 = MT_NULL, const MultiType& param1 = MT_NULL,
             const MultiType& param2 = MT_NULL, const MultiType& param3 = MT_NULL,
             const MultiType& param4 = MT_NULL);

    void cloning(Executor* executor) const;

  protected:
    short                       functorType;      //!< The type of Function we've got (either static or objective).
    unsigned int                paramCount;       //!< the count of parameters.
    MultiType                   defaultValue[5];  //!< Default Values.
};

#include "executor/executor_functional.h"
#define EXECUTOR_FUNCTIONAL_USE_CONST
#include "executor/executor_functional.h"
#define EXECUTOR_FUNCTIONAL_USE_STATIC
#include "executor/executor_functional.h"

#endif /* _EXECUTOR_H */
