/*!
 * @file class_list.h
  *  Definition of the Class List, that handles a Class-Specific-Control structure

 */

#ifndef _CLASS_LIST_H
#define _CLASS_LIST_H

#include "class_id.h"
#include <list>
#include <string>

// FORWARD DECLARATION
class BaseObject;

//! A class that handles Pointers to Objects of all type.
/**
 * here all the Pointers to all the Object of orxonox are stored, that implement BaseObject
 * for now.
 * You can get Any Object's Reference to BaseObject with dynamic_cast<T>(ClassList::getObject(name, CL_T_NAME));
 *  where: T: is the Class to cast to,
 *   name: the name of the Object (not className)
 *   CL_T_NAME: the class Identifier, (if CL_NULL or nothing it will take longer, because all BaseObject's are searched through)
 *
 * There is also the exists-function, that just checks, if a Reference is still in existence.
 *
 * @see ClassID, BaseObject, dynamic_cast
 */
class ClassList {
  public:
    ClassList(ClassID classID, unsigned long classIDFull, const std::string& className);
    virtual ~ClassList();

    /* MAINTENANCE FUNCTIONS THESE ARE !!ONLY FOR BASEOBJECT !! */
    static ClassList*                     addToClassList(BaseObject* objectPointer, ClassID classID, unsigned long classIDFull, const std::string& className);
    static void                           removeFromClassList(BaseObject* objectPointer);



    static const std::list<BaseObject*>*  getList(ClassID classID = CL_NULL);// { return (ClassList* fl = ClassList::getClassList(classID) != NULL)? &(fl->objectList) : NULL; };
    static const std::list<BaseObject*>*  getList(const std::string& className); // { return (ClassList* fl = ClassList::getClassList(className) != NULL)? &(fl->objectList) : NULL;  };
    static const std::list<std::string>*  getClassNames();
    static BaseObject*                    getObject(const std::string& objectName, ClassID classID = CL_NULL);
    static BaseObject*                    getObject(const std::string& objectName, const std::string& className);
    static bool                           exists(const BaseObject* object, ClassID classID = CL_NULL);
    static bool                           exists(const std::string& className, const std::string& objectName);

    void                                  sendBack(std::list<BaseObject*>::const_iterator it);

    static void                           whatIs(const BaseObject* object);

    static const std::string&             IDToString(ClassID classID = CL_NULL);
    static ClassID                        StringToID(const std::string& className);
    static void                           debug(unsigned int debugLevel = 0, ClassID classID = CL_NULL);
    static void                           debugS(const std::string& className = "", unsigned int debugLevel = 0);

    inline bool                           operator==(ClassID classID) { return (this->classID == classID); };
    bool                                  operator==(const std::string& className);
    inline ClassID                        getLeafClassID() const { return this->classID; };

  private:
    static ClassList*                     getClassList(ClassID classID);
    static ClassList*                     getClassList(const std::string& className);

  private:
    ClassID                               classID;                //!< ClassID stored in this ClassList
    unsigned long                         classIDFull;            //!< The Full ClassID of this Class.

    const std::string                     className;              //!< Name of the Class Stored here

    std::list<BaseObject*>                objectList;             //!< A list of Objects belonging to this Class

    // STATIC MEMBERS
    static std::list<ClassList>*          classList;              //!< The first Class in the List
    static std::list<std::string>         classNames;             //!< a List of all Names of all classes, that have registered so far.
};

#endif /* _CLASS_LIST_H */
