/*!
 * @file collision.h
 *  Definition of a collision as a two WE hit each other
 *
 *  A is shared between two WorldEntity's CollisionHandles if both are subscribed to this event. In this case only one
 *  of the two CollisionHandles will calculate the CollisionReaction and the bDispatched flag will be set afterwards
 *  to signal that it's already cared about and should be ignored.
 */

#ifndef _COLLISION_H
#define _COLLISION_H

#include "vector.h"
#include <vector>

class WorldEntity;
class BoundingVolume;
class CollisionEvent;

//! A class representing a simple collision
class Collision
{

  public:
    Collision();
    virtual ~Collision();

    /** collides two WorldEntities @param entityA world entity A, @param entityB world entity B, @param bvA volume A @param bvB volumeB */
    inline void collide(WorldEntity* entityA, WorldEntity* entityB) { this->entityA = entityA; this->entityB = entityB; this->bDispatched = false; }


    /** @return Collision WorldEntity A */
    inline WorldEntity* getEntityA() const { return this->entityA; }
    /** @return Collision WorldEntity B */
    inline WorldEntity* getEntityB() const { return this->entityB; }
    /** @return true if Entity A collides */
    inline bool isEntityACollide() const { return this->entityACollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityA*/
    inline void setEntityACollide(bool flag) { this->entityACollide = flag; }
    /** @return true if Entity B collides */
    inline bool isEntityBCollide() const { return this->entityBCollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityB*/
    inline void setEntityBCollide(bool flag) { this->entityACollide = flag; }


    /** @returns true if this Collision has already been dispatched */
    inline bool isDispatched() { return this->bDispatched; }
    /** sets the dispatched flag to true */
    inline void dispatched() { this->bDispatched = true; }

    /** registers a @param event CollisionEvent to take place */
    inline void registerCollisionEvent(CollisionEvent* event) { this->collisionEvents.push_back(event); this->bDispatched = false;}
    /** @returns a vector of collision events */
    inline const std::vector<CollisionEvent*>& getCollisionEvents() const { return this->collisionEvents; }


    void flushCollisionEvents();


  private:
    WorldEntity*                 entityA;                       //!< the collision body A
    WorldEntity*                 entityB;                       //!< the collision body B
    bool                         entityACollide;                //!< true if entity A is subscribed for collision reaction
    bool                         entityBCollide;                //!< true if entity B is subscribed for collision reaction

    bool                         bDispatched;                   //!< true if this collision has already been dispatched

    std::vector<CollisionEvent*> collisionEvents;               //!< the collision event list
};

#endif /* _COLLISION_H */
