/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "texture.h"

#include "debug.h"
#include "compiler.h"
#include <math.h>

#ifdef HAVE_SDL_SDL_H
#include <SDL/SDL_image.h>
#include <SDL/SDL_endian.h>
#include <SDL/SDL_byteorder.h>
#else
#include <SDL_endian.h>
#include <SDL_image.h>
#include <SDL_byteorder.h>
#endif
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
/*
 * On the BIG_ENDIAN architecture, the 24 and 32bit bitmaps have
 * different masks. If you don't do this distinction properly, 
 * you will get weird-looking textures.
 */
Uint32 alphaMask[] = {
	0xFF000000,
	0x00FF0000,
	0x0000FF00,
	0x000000FF,
};

Uint32 opaqueMask[] = {
	0x00FF0000,
	0x0000FF00,
	0x000000FF,
	0xFF000000
};
#else
/*
 * On the LIL_ENDIAN architecture everything is fine and easy. The 24
 * and 32bit bitmaps have the same masks.
 */
Uint32 alphaMask[] = {
	0x000000FF,
	0x0000FF00,
	0x00FF0000,
	0xFF000000,
};

Uint32 opaqueMask[] = alphaMask

#endif
/**
 * @brief Constructor for a Texture
*/
Texture::Texture(const std::string& imageName, GLenum target)
{
  this->setClassID(CL_TEXTURE, "Texture");

  this->bAlpha = false;
  this->texture = 0;
  this->image = NULL;
  this->priority = 0.5;

  if (!imageName.empty())
  {
    this->setName(imageName);
    this->loadImage(imageName, target);
  }
}


/**
 * @brief Destructor of a Texture
 *
 * Frees Data, and deletes the textures from GL
 */
Texture::~Texture()
{
  if (this->texture != 0)
    glDeleteTextures(1, &this->texture);
  if (this->image != NULL)
    SDL_FreeSurface(this->image);
}


/**
 * @brief loads an Image from a file to a Texture
 * @param imageName The image to load
*/
bool Texture::loadImage(const std::string& imageName, GLenum target)
{
  if (Texture::texturesEnabled)
  {
    if (this->image != NULL)
    {
      SDL_FreeSurface(this->image);
      this->image = NULL;
    }
    if (this->texture != 0)
    {
      glDeleteTextures(1, &this->texture);
      this->texture = 0;
    }
    if (!imageName.empty())
    {
      SDL_Surface* tmpSurf;
      if (this->texture != 0 && glIsTexture(this->texture))
        glDeleteTextures(1, &this->texture);
      // load the new Image to memory
      tmpSurf = IMG_Load(imageName.c_str());
      if(tmpSurf != NULL)
      {
        PRINTF(4)("loading Image %s\n", imageName.c_str());
        bool hasAlpha;
        SDL_Surface* newSurf = this->prepareSurface(tmpSurf, hasAlpha);
        if (newSurf != NULL)
        {
          this->setSurface(newSurf);
          this->setAlpha(hasAlpha);
          this->setTexture(Texture::loadTexToGL(newSurf, target));
        }

        SDL_FreeSurface(tmpSurf);
        return true;
      }
      else
      {
        PRINTF(1)("IMG_Load: %s\n", IMG_GetError());
        this->texture = 0;
        return false;
      }
    }
    else
    {
      PRINTF(2)("Image-Name not specified\n");
      return false;
    }
  }
  return false;
}


/**
 * @brief rebuilds the texture.
 *
 * reloads the Texture from Memory to OpenGL.
 */
bool Texture::rebuild()
{
  if (this->texture != 0)
  {
    if (glIsTexture(this->texture))
      glDeleteTextures(1,&this->texture);
    this->setTexture(0);
  }

  if (this->image != NULL)
  {
    PRINTF(3)("Reloading Texture of %s '%s'\n", this->getClassName(), this->getName());
    this->setTexture(loadTexToGL(this->image));
  }
}


/**
 * @brief set the surface this Texture handles
 * @param newSurface the new Surface to set as the image for this Texture.
 *
 * This deletes the old version of the stored Texture,
 * and sets the newly given Surface as current.
 */
bool Texture::setSurface(SDL_Surface* newSurface)
{
  if (this->image != NULL)
    SDL_FreeSurface(this->image);

  this->image = newSurface;

  return (this->image != NULL);
}


bool Texture::texturesEnabled = true;

/**
 * @brief enables, disables textures
 * @param texturesEnabled true if the textures should be enabled
 */
void Texture::setTextureEnableState(bool texturesEnabled)
{
  Texture::texturesEnabled = texturesEnabled;
}


//////////////////////////////////////
// UTILITY FUNCTIONALITY OF TEXTURE //
//////////////////////////////////////
/**
 * @brief converts surface to a new SDL_Surface, that is loadable by openGL
 * @param surface the Surface to convert
 * @param hasAlpha if the newly created Surface has an alpha channel, true is returned otherwise false.
 * @returns a !!new!! Surface, that is loadable by openGL.
 */
SDL_Surface* Texture::prepareSurface(SDL_Surface* surface, bool& hasAlpha) const
{
  assert(surface != NULL);
  PRINTF(4)("Loading texture to OpenGL-Environment.\n");

  SDL_Surface* retSurface;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;
  hasAlpha = false;
  int pixelDepth = 24;

	Uint32* mask = opaqueMask;
	
  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA | SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( saved_flags & SDL_SRCALPHA )
  {
    SDL_SetAlpha(surface, 0, 0);
    hasAlpha = true;
    pixelDepth = 32;
		mask = alphaMask;
  }
		
  retSurface = SDL_CreateRGBSurface(SDL_HWSURFACE,
                                    surface->w, surface->h,
                                    pixelDepth,
																		mask[0], mask[1], mask[2], mask[3] );
  if ( retSurface == NULL )
    return NULL;

  /* Copy the surface into the GL texture image */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, retSurface, &area);

  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA )
  {
    SDL_SetAlpha(surface, saved_flags | SDL_OPENGL, saved_alpha);
    hasAlpha = true;
  }

  return (retSurface);
}


/**
 * @brief Loads a Texture to the openGL-environment.
 * @param surface the Image to load to openGL
 * @returns The ID of the texture.
 */
GLuint Texture::loadTexToGL (const SDL_Surface* surface, GLenum target) const
{
  //   if (this->texture != 0 && glIsTexture(this->texture))
  //     glDeleteTextures(1, &this->texture);
  //   this->texture = 0;
  assert(surface != NULL);

  int      errorCode = 0;           //!< the error code for the texture loading functions
  GLuint   texture;                 //!< the OpenGL texture handle
  int      mipmapLevel = 0;         //!< the maximum mipmap level for this texture
  int      mipmapWidth = 0;         //!< the width of the mipmap
  int      mipmapHight = 0;         //!< the height of the mipmap
  GLenum   format = GL_RGB;
  if (this->bAlpha)
  {
    format = GL_RGBA;
    assert(surface->format->BitsPerPixel == 32);
  }
  else
  {
    assert(surface->format->BitsPerPixel == 24);
  }

  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(target, texture);

//   glTexImage2D(target,  0,  format,
//                surface->w,  surface->h,
//                0, format,  GL_UNSIGNED_BYTE,
//                surface->pixels);

  glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(target, GL_TEXTURE_WRAP_R, GL_REPEAT);

  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR/*_MIPMAP_LINEAR*/);
  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_PRIORITY, this->priority);


  /* control the mipmap levels */
  glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_MIN_LOD, 5);
  glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_MAX_LOD, 0);

  /* build the Texture  OpenGL V >= 1.1 */

  //  printf("%s, w:%d h:%d, 0x%x\n", this->getName(), surface->w, surface->h, target);

  // build the MipMaps automaticaly
  errorCode = gluBuild2DMipmaps(target, format,
                                surface->w,  surface->h,
                                format,  GL_UNSIGNED_BYTE,
                                surface->pixels
                               );
  if(unlikely(errorCode != 0))
    PRINTF(1)("Error while loading texture (mipmap generation), gluBuild2DMipmaps returned %i\n", errorCode);

  return texture;
}

