/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/

/*!
 * @file vector.h
 * A basic 3D math framework
 *
 * Contains classes to handle vectors, lines, rotations and planes
*/

#ifndef __VECTOR_H_
#define __VECTOR_H_

#include <math.h>
#include "compiler.h"
//! PI the circle-constant
#define PI 3.14159265359f


//! this is a small and performant 3D vector
typedef float sVec3D[3];


//! small and performant 2D vector
typedef float sVec2D[2];



//! 3D Vector
/**
        Class to handle 3D Vectors
*/
class Vector {
 public:
  Vector (float x, float y, float z) : x(x), y(y), z(z) {}  //!< assignment constructor
  Vector () : x(0), y(0), z(0) {}
  ~Vector () {}

  /** @param v: the Vecor to compare with this one @returns true, if the Vecors are the same, false otherwise */
  inline bool operator== (const Vector& v) const { return (this->x==v.x&&this->y==v.y&&this->z==v.z)?true:false; };
  /** @param index The index of the "array" @returns the x/y/z coordinate */
  inline float operator[] (float index) const {if( index == 0) return this->x; if( index == 1) return this->y; if( index == 2) return this->z; }
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector operator+ (const Vector& v) const { return Vector(x + v.x, y + v.y, z + v.z); };
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector operator+ (const sVec3D& v) const { return Vector(x + v[0], y + v[1], z + v[2]); };
  /** @param v The vector to add  @returns the addition between two vectors (this += v) */
  inline const Vector& operator+= (const Vector& v) { this->x += v.x; this->y += v.y; this->z += v.z; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline const Vector& operator+= (const sVec3D& v) { this->x += v[0]; this->y += v[1]; this->z += v[2]; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector operator- (const Vector& v) const { return Vector(x - v.x, y - v.y, z - v.z); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector operator- (const sVec3D& v) const { return Vector(x - v[0], y - v[1], z - v[2]); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector& operator-= (const Vector& v) { this->x -= v.x; this->y -= v.y; this->z -= v.z; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector& operator-= (const sVec3D& v) { this->x -= v[0]; this->y -= v[1]; this->z -= v[2]; return *this; };
  /** @param v the second vector  @returns The dotProduct between two vector (this (dot) v) */
  inline float operator* (const Vector& v) const { return x * v.x + y * v.y + z * v.z; };
  /** @todo strange */
  inline const Vector& operator*= (const Vector& v) { this->x *= v.x; this->y *= v.y; this->z *= v.z; return *this; };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this * f) */
  inline Vector operator* (float f) const { return Vector(x * f, y * f, z * f); };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this *= f) */
  inline const Vector& operator*= (float f) { this->x *= f; this->y *= f; this->z *= f; return *this; };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this / f) */
  inline Vector operator/ (float f) const { return (unlikely(f == 0.0))?Vector(0,0,0):Vector(this->x / f, this->y / f, this->z / f); };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this /= f) */
  inline const Vector& operator/= (float f) {if (unlikely(f == 0.0)) {this->x=0;this->y=0;this->z=0;} else {this->x /= f; this->y /= f; this->z /= f;} return *this; };
  /**  copy constructor @todo (i do not know it this is faster) @param v the vector to assign to this vector. @returns the vector v */
  inline const Vector& operator= (const Vector& v) { this->x = v.x; this->y = v.y; this->z = v.z; return *this; };
  /** copy constructor* @param v the sVec3D to assign to this vector. @returns the vector v */
  inline const Vector& operator= (const sVec3D& v) { this->x = v[0]; this->y = v[1]; this->z = v[2]; }
  inline const Vector& operator= (const float* v) { this->x = v[0]; this->y = v[1]; this->z = v[2]; }
  /** @param v: the other vector \return the dot product of the vectors */
  float dot (const Vector& v) const { return x*v.x+y*v.y+z*v.z; };
  /** @param v: the corss-product partner @returns the cross-product between this and v (this (x) v) */
  inline Vector cross (const Vector& v) const { return Vector(y * v.z - z * v.y, z * v.x - x * v.z, x * v.y - y * v.x ); }
  /** scales the this vector with v* @param v the vector to scale this with */
  void scale(const Vector& v) {   x *= v.x;  y *= v.y; z *= v.z; };
  /** @returns the length of the vector */
  inline float len() const { return sqrt (x*x+y*y+z*z); }
  /** normalizes the vector */
  inline void normalize() { float l = len(); if( unlikely(l == 0.0))return; this->x=this->x/l; this->y=this->y/l; this->z=this->z/l; };
  Vector getNormalized() const;
  Vector abs();

  /** @param toVec nears this Vector to... @param t how much to advance (0.0: not at all; 1.0: fully) */
  inline void slerpTo(const Vector& toVec, float t) { *this + (toVec - *this) * t; };

  void debug() const;

 public:
  float    x;     //!< The x Coordinate of the Vector.
  float    y;     //!< The y Coordinate of the Vector.
  float    z;     //!< The z Coordinate of the Vector.
};

/**
 *  calculate the angle between two vectors in radiances
 * @param v1: a vector
 * @param v2: another vector
 * @return the angle between the vectors in radians
*/
inline float angleDeg (const Vector& v1, const Vector& v2) { return acos( v1 * v2 / (v1.len() * v2.len())); };
/**
 *  calculate the angle between two vectors in degrees
 * @param v1: a vector
 * @param v2: another vector
 * @return the angle between the vectors in degrees
*/
inline float angleRad (const Vector& v1, const Vector& v2) { return acos( v1 * v2 / (v1.len() * v2.len())) * 180/M_PI; };

/** an easy way to create a Random Vector @param sideLength the length of the Vector (x not sqrt(x^2...)) */
#define VECTOR_RAND(sideLength)  (Vector((float)rand()/RAND_MAX -.5, (float)rand()/RAND_MAX -.5, (float)rand()/RAND_MAX -.5) * sideLength)

#endif /* __VECTOR_H_ */

