

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Benjamin GRauer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOAD

#include "glmenu_imagescreen.h"

#include "graphics_engine.h"
#include "material.h"
#include "util/loading/factory.h"
#include "util/loading/load_param.h"

CREATE_FACTORY(GLMenuImageScreen, CL_GLMENU_IMAGE_SCREEN);

using namespace std;
/**
 * @param root The Element to load the GLMenu from
 */
GLMenuImageScreen::GLMenuImageScreen(const TiXmlElement* root)
{
  this->setClassID(CL_GLMENU_IMAGE_SCREEN, "GLMenuImageScreen");
  this->setName("GLMenuLoadScreen");
  // Select Our VU Meter Background Texture
  this->backMat = new Material("load_screen");
  this->barMat = new Material("bar");
  this->barMat->setDiffuse(1,1,1);
  this->backMat->setDiffuse(1,1,1);
  this->maxValue = 10;
  this->currentValue = 0;
  this->setPosition(0,0);
  this->setScale(1,1);
  this->setBarPosScale( .6, .75, .3, .1);
  // End of Background image code.

  if (root != NULL)
    this->loadParams(root);
}

/**
 *  Loads a GLMenu from an inputElement
 * @param root The Element to load the GLMenu from
*/
void GLMenuImageScreen::loadParams(const TiXmlElement* root)
{
  LoadParam(root, "BackgroundImage", this, GLMenuImageScreen, setBackgroundImage)
    .describe("sets the image to load onto the loadscreen");

  LoadParam(root, "BackgroundPS", this, GLMenuImageScreen, setPosScale)
    .describe("The Position and Scale of the Background Image in %(0-1.0). PosX, PosY, SizeX, SizeY");

  LoadParam(root, "BarImage", this, GLMenuImageScreen, setBarImage)
    .describe("sets the image of the LoadingBar");

  LoadParam(root, "BarPS", this, GLMenuImageScreen, setBarPosScale)
    .describe("The Position and Scale of the Loading Bar in %(0-1.0). PosX, PosY, SizeX, SizeY");

  LoadParam(root, "ElementCount", this, GLMenuImageScreen, setMaximum)
    .describe("The Count of elements to load into the bar (this is only a maximum value)");
}

/**
 *  standard deconstructor
   @todo this deconstructor is not jet implemented - do it
*/
GLMenuImageScreen::~GLMenuImageScreen()
{
  delete this->backMat;
  delete this->barMat;
}

/**
  * @brief sets the background image name
  * @param backImageName name of the backgroun-image
 */
void GLMenuImageScreen::setBackgroundImage (const std::string& backImageName)
{
  this->backMat->setDiffuseMap(backImageName);
}

/**
 * @brief sets position of the ImageScreen
 * @param offsetX offset from the top left corner in percent(0-1) of the screensize
 * @param offsetY offset from the top left corner in percent(0-1) of the screensize
*/
void GLMenuImageScreen::setPosition(float offsetX, float offsetY)
{
  this->offsetX = offsetX;
  this->offsetY = offsetY;
}

/**
 * @brief sets size of the ImageScreen
 * @param scaleX the scaleing of the image into the x-direction (in percent (0-1))
 * @param scaleY the scaleing of the image into the y-direction (in percent (0-1))
*/
void GLMenuImageScreen::setScale(float scaleX, float scaleY)
{
  this->scaleX = scaleX;
  this->scaleY = scaleY;
}

/**
 * @brief sets position and size of the ImageScreen
 * @param offsetX offset from the top left corner in percent(0-1) of the screensize
 * @param offsetY offset from the top left corner in percent(0-1) of the screensize
 * @param scaleX the scaleing of the image into the x-direction (in percent (0-1))
 * @param scaleY the scaleing of the image into the y-direction (in percent (0-1))
*/
void GLMenuImageScreen::setPosScale(float offsetX, float offsetY, float scaleX, float scaleY)
{
  this->setPosition(offsetX, offsetY);
  this->setScale(scaleX, scaleY);
}

/**
 * @param barImage An image for the Bar
*/
void GLMenuImageScreen::setBarImage(const std::string& barImage)
{
  this->barMat->setDiffuseMap(barImage);
}

/**
 * @brief sets the Position and the Size of the bar
 * @param barX The Position in the x-direction in percent of the screen (0-1)
 * @param barY The Position in the y-direction in percent of the screen (0-1)
 * @param barW The Size in the x-direction in percent of the screen (0-1)
 * @param barH The Size in the y-direction in percent of the screen (0-1)
*/
void GLMenuImageScreen::setBarPosScale(float barX, float barY, float barW, float barH)
{
  this->barX = barX;
  this->barY = barY;
  this->barW = barW;
  this->barH = barH;
}


/**
 * @brief set the maximum of countable steps
 * @param maxValue of steps
*/
void GLMenuImageScreen::setMaximum(int maxValue)
{
  this->maxValue = maxValue;
}

/**
 * @brief set current value
 * @param currentValue value to set
*/
void GLMenuImageScreen::setValue(int currentValue)
{
  this->currentValue = currentValue;
  this->draw();
}


/**
 * @brief get the current value
 */
int GLMenuImageScreen::getValue()
{
  return this->currentValue;
}


/**
 * @brief call this to trigger a progress event
 *
 * this has to redraw the progress bar and the whole image
 */
void GLMenuImageScreen::step ()
{
  if (this->currentValue < this->maxValue)
  {
    this->currentValue++;
    this->draw();
  }
  else
    PRINTF(2)("ImageScreen-loadbar exceeds maximum value %d\n", this->maxValue);
}



/**
 * @brief draws the ImageScreen to the screenbuffer
*/
void GLMenuImageScreen::draw ()
{
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

  PRINTF(4)("GLMenuImagEscreen::draw() - drawing step %i/%i\n",
            this->currentValue, this->maxValue);

  /* screen size */
  int screenWidth = GraphicsEngine::getInstance()->getResolutionX();
  int screenHeight = GraphicsEngine::getInstance()->getResolutionY();

  int imageWidth = (int)(screenWidth * this->scaleX);
  int imageHeight = (int)(screenHeight * this->scaleY);

  int offsetX = (int)(this->offsetX * screenWidth);
  int offsetY = (int)(this->offsetY * screenHeight);

  /* loadbar pos */
  int barX = (int)(this->barX *screenWidth);
  int barY = (int)(this->barY *screenHeight);
  int barW = (int)(this->barW *screenWidth);
  int barH = (int)(this->barH *screenHeight);

  float val = (float)this->currentValue/(float)this->maxValue;

  if( val > barW)
    val = barW;

  GraphicsEngine::enter2DMode();

  /* draw the BackGround */
  backMat->select();
  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2i(0, 1); glVertex2i(offsetX, offsetY + imageHeight);
  glTexCoord2i(1, 1); glVertex2i(offsetX +imageWidth, offsetY + imageHeight);
  glTexCoord2i(0, 0); glVertex2i(offsetX, offsetY);
  glTexCoord2i(1, 0); glVertex2i(offsetX + imageWidth, offsetY);
  glEnd();

  /* draw white border */
  glBegin(GL_LINE_LOOP);
  glColor3f(1.0, 1.0, 1.0);
  glVertex2i(barX - 2, barY - 2);
  glVertex2i(barX + barW + 2, barY - 2);
  glVertex2i(barX + barW + 2, barY + barH + 2);
  glVertex2i(barX - 2, barY + barH + 2);
  glColor3f(1.0, 1.0, 1.0);
  glEnd();

  /* draw the progress bar */
  barMat->select();
  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2f(0,   1); glVertex2i(barX, barY + barH);
  glTexCoord2f(val, 1); glVertex2i(barX + (int)(val * this->barW * (float)screenWidth), barY + barH);
  glTexCoord2f(0,   0); glVertex2i(barX, barY);
  glTexCoord2f(val, 0); glVertex2i(barX + (int)(val * this->barW * (float)screenWidth), barY);
  glEnd();

  /*
    glBegin(GL_QUADS);
    glColor3f(0.0, 0.0, 0.0);
    glVertex2i(barX, barY);
    glVertex2i(barX + barWidth, barY);
    glVertex2i(barX + barWidth, barY + barHeight);
    glVertex2i(barX, barY + barHeight);
    glColor3f(1.0, 1.0, 1.0);
    glEnd();

    /* draw black border
    glBegin(GL_QUADS);
    glColor3f(0.0, 0.0, 0.0);
    glVertex2i(barX-1, barY-1);
    glVertex2i(barX + barWidth +1, barY-1);
    glVertex2i(barX + barWidth+1, barY + barHeight+1);
    glVertex2i(barX - 1, barY + barHeight +1);
    glColor3f(1.0, 1.0, 1.0);
    glEnd();

  */

  GraphicsEngine::leave2DMode();

  SDL_GL_SwapBuffers();
}
