/*!
 * @file cr_engine.h
 * @brief The collision reaction engine, defining generic collision reactions to collision events
*/

#ifndef _CR_ENGINE_
#define _CR_ENGINE_

#include "base_object.h"
#include <stdarg.h>
#include <vector>

// FORWARD DECLARATION
class CollisionHandle;
class Collision;
class WorldEntity;

//! A default singleton class.
class CREngine : public BaseObject
{

  public:
  typedef enum CRType {
    CR_PHYSICS_MOMENTUM   = 0,
    CR_PHYSICS_GROUND,
    CR_PHYSICS_GROUND_WALK,

    CR_OBJECT_DAMAGE,
    CR_OBJECT_PICKUP,

    CR_VERTEX_TRAFO,

    CR_SPECIAL_CALLBACK,

    CR_NUMBER
  };

  virtual ~CREngine(void);

  /** @returns a Pointer to the only object of this Class */
  inline static CREngine* getInstance(void) { if (!singletonRef) singletonRef = new CREngine();  return singletonRef; };


  CollisionHandle* subscribeReaction(WorldEntity* worldEntity, CRType type, int nrOfTargets, ...);

  bool unsubscribeReaction(WorldEntity* worldEntity);
  bool unsubscribeReaction(CollisionHandle* collisionHandle);


  void handleCollisions();

  /** @returns an instance to a collision object. instead of creating new object this ones can be resycled */
  inline Collision* getCollisionObject() { /* return the first element of the cache list*/ }
  /** @param collision: returns the Collision object back to the cache list */
  inline void putCollisionObject(Collision* collision) { this->cachedCollisions.push_back(collision); }


private:
  CREngine(void);


private:
  std::vector<CollisionHandle*>       collisionHandles;         //!< list with the collision handles
  std::vector<Collision*>             cachedCollisions;         //!< a list of unused, cached collision events

  static CREngine*                    singletonRef;             //!< the reference to the CREngine object (singleton)
};

#endif /* _CR_ENGINE_ */
