/*!
 * @file shell_buffer.h
 * @brief The Shell buffer Tasks
 * @see debug.h
*/

#ifndef _SHELL_BUFFER_H
#define _SHELL_BUFFER_H

#include <string>
#include <list>
#include <stdarg.h>

#define      SHELL_BUFFER_SIZE       16384         //!< The Size of the input-buffers (should be large enough to carry any kind of input)

namespace OrxShell
{
  //! A class handling output from orxonox via debug.h
  /**
   * the ShellBuffer redirects output from PRINTF(x) to the Shell and STDOUT
   * the ShellBuffer is a front-filling queue of limited length, that has the
   * youngest added Entry at the beginning, and the oldest at the end.
   */
  class ShellBuffer
  {

  public:
    virtual ~ShellBuffer();
    /** @returns a Pointer to the only object of this Class */
    inline static ShellBuffer* getInstance() { if (!ShellBuffer::singletonRef) ShellBuffer::singletonRef = new ShellBuffer();  return ShellBuffer::singletonRef; };
    /** @returns true if this class is instanciated, false otherwise */
    inline static bool isInstanciated() { return (ShellBuffer::singletonRef == NULL)?false:true; };

    static void addBufferLineStatic(const char* line, ...);
    void addBufferLine(const char* line);

    /// BUFFER
    /** @param bufferSize the new Buffer-Size */
    void setMaxBufferSize(unsigned int maxBufferSize) { this->maxBufferSize = maxBufferSize; };
    void flush();

    /** @returns the List of stings from the Buffer */
    const std::list<std::string>& getBuffer() const { return this->buffer; };
    /** @returns the Count of lines processed by the Shell. */
    inline unsigned long getLineCount() const { return this->lineCount; };
    /** @returns the Current Buffer Size. */
    inline unsigned int getBufferSize() const { return this->buffer.size(); };

    void debug() const;

  private:
    ShellBuffer();

  private:
    static ShellBuffer*           singletonRef;                       //!< The singleton-reference to the only memeber of this class.
    unsigned int                  maxBufferSize;                         //!< The Size of the buffer

    std::string                   keepBuffer;                         //!< a BUFFER to have multi-non-newLine commands be copied into the shell.

    unsigned long                 lineCount;                          //!< how many Lines have been written out so far.

    // The Beginning of buffer (buffer.front()) is the last added line.
    static char                   bufferArray[SHELL_BUFFER_SIZE];     //!< a BUFFER for fast writing
    static std::list<std::string> buffer;                             //!< A list of stored char-arrays(strings) to store the history
  };

}

#endif /* _SHELL_BUFFER_H */
