/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GUI

#include "glgui_inputline.h"

namespace OrxGui
{
  /**
   * @brief standard constructor
  */
  GLGuiInputLine::GLGuiInputLine ()
  {
    this->init();
  }


  /**
   * @brief standard deconstructor
   */
  GLGuiInputLine::~GLGuiInputLine()
  {}

  float GLGuiInputLine::repeatDelay = 0.3f;
  float GLGuiInputLine::repeatRate  = 0.01f;


  /**
   * @brief initializes the GUI-element
   */
  void GLGuiInputLine::init()
  {
    this->setClassID(CL_GLGUI_INPUTLINE, "GLGuiInputLine");

    this->setFocusable(true);

    this->text.setParent2D(this);
    this->text.setRelCoor2D(4,4);
    this->text.setFont("fonts/final_frontier.ttf", 20);
    this->text.setVisibility(false);
    this->resize();

  }


  /**
   * @brief sets the Text of the InputLine
   * @param text The new Text.
   */
  void GLGuiInputLine::setText(const std::string& text)
  {
    this->text.setText(text);
    this->resize();

    emit(this->textChanged(this->getText()));
  }

  /**
   * @brief appends text to the InputLine
   * @param appendText the Text to append
   */
  void GLGuiInputLine::append(const std::string& appendText)
  {
    this->text.append(appendText);
    this->resize();
    emit(this->textChanged(this->text.getText()));
  }


  /**
   * @brief appends a Character to the InputLine
   * @param character the Character to append.
   */
  void GLGuiInputLine::appendCharacter(char character)
  {
    this->text.appendCharacter(character);
    this->resize();
    emit(this->textChanged(this->text.getText()));
  }


  /**
   * @brief Removes Characters from the InputLine
   * @param chars The count of characters to remove
   */
  void GLGuiInputLine::removeCharacters(unsigned int chars)
  {
    this->text.removeCharacters(chars);
    this->resize();
    emit(this->textChanged(this->text.getText()));
  }


  /**
   * removes the focus from this Widget.
   */
  void GLGuiInputLine::removedFocus()
  {
    GLGuiWidget::removedFocus();
    this->pressedKey = 0;
    this->pressedKeyName = 0;
  }


  /**
   * Processes an Event.
   * @param event The event to be processed
   * @return true if the event was catched.
   */
  bool GLGuiInputLine::processEvent(const Event& event)
  {
    if (event.bPressed)
    {
      if (event.type == SDLK_BACKSPACE)
      {
        this->delayNext = GLGuiInputLine::repeatDelay;
        this->pressedKey = SDLK_BACKSPACE;
        this->pressedKeyName = SDLK_BACKSPACE;
        this->removeCharacters(1);
        return true;
      }
      // any other keyboard key
      else if (likely(event.type < 127))
      {
        this->delayNext = GLGuiInputLine::repeatDelay;

        this->appendCharacter(event.x);
        this->pressedKey = event.type;
        this->pressedKeyName = event.x;
        return true;
      }
    }
    else // if(!event.bPressed)
    {
      if (this->pressedKey == event.type)
      {
        this->pressedKeyName = 0;
        this->pressedKey = 0;
        this->delayNext = 0.0;
        return true;
      }
    }

    return false;
  }


  /**
   * @brief Resizes the Widget to the new Size-constraints.
   */
  void GLGuiInputLine::resize()
  {
    this->text.setRelCoor2D(this->borderLeft() + 2.0,this->borderTop() + 2.0);
    this->setSize2D( this->text.getSize2D() + Vector2D(borderLeft() + borderRight() + 4.0, borderTop() + borderBottom() + 4.0));
    GLGuiWidget::resize();
    this->frontRect().setTopLeft(borderLeft(), borderTop());
    this->frontRect().setSize(this->getSize2D() - Vector2D(borderLeft() + borderRight(), borderTop() + borderBottom()));
  }


  void GLGuiInputLine::hiding()
  {
    this->text.setVisibility(false);
  }

  void GLGuiInputLine::showing()
  {
    this->text.setVisibility(true);
  }


  /**
   * ticks the InputLine
   * @param dt the time passed.
   */
  void GLGuiInputLine::tick(float dt)
  {
    if (this->delayNext > 0.0)
      this->delayNext -= dt;
        else if (this->pressedKey != SDLK_FIRST )
    {
      this->delayNext = GLGuiInputLine::repeatRate;
      switch (this->pressedKey)
      {
        case SDLK_BACKSPACE:
          this->removeCharacters(1);
          break;
        default:
        {
          if (likely(this->pressedKey < 127))
            this->appendCharacter(this->pressedKeyName);
        }
      }
    }


  }

  /**
   * @brief draws the GLGuiInputLine
   */
  void GLGuiInputLine::draw() const
  {
    this->beginDraw();
    GLGuiWidget::draw();

    this->frontMaterial().select();
    GLGuiWidget::drawRect(this->frontRect());

    this->endDraw();
  }
}
