/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_COLLISION_REACTION

#include "collision_handle.h"

#include "world_entity.h"

#include "collision.h"
#include "collision_event.h"

using namespace std;


/**
 * standard constructor
 * @todo this constructor is not jet implemented - do it
*/
CollisionHandle::CollisionHandle (WorldEntity* owner, CREngine::CRType type)
{
  this->setClassID(CL_COLLISION_HANDLE, "CollisionHandle");

  this->owner = owner;
  this->type = type;

  this->bCollided = false;
  this->bContinuousPoll = false;
  this->bDispatched = false;
  this->bStopOnFirstCollision = false;
}


/**
 * standard deconstructor
*/
CollisionHandle::~CollisionHandle ()
{
  // delete what has to be deleted here
}

/**
 * restores the CollisionHandle to its initial state
 */
void CollisionHandle::reset()
{
  this->flushCollisions();
}


/**
 * add more filter targets to this collision handle
 *  @param classID the classid to look for
 */
void CollisionHandle::addTarget(long target)
{
  // make sure there is no dublicate
  std::vector<long>::iterator it = this->targetList.begin();
  for( ; it < this->targetList.end(); it++)
    if( (*it) == target)
      return;

  // add element
   PRINTF(0)("addTarget: %i \n", target);
   this->targetList.push_back(target);
}


/**
 * registers a new Collision Object
 *  @param entityA WorldEntity A of the collision
 *  @param entityB WorldEntity B of the collision
 * if a there is already a collision object with the same stats
 * registration will be skipped and the last collision object is returned
 */
Collision* CollisionHandle::registerCollision(WorldEntity* entityA, WorldEntity* entityB)
{
  //first get the collision object, multiple sources
  Collision* c;
  if( this->collisionList.empty() ||
      ((this->collisionList.back())->getEntityA() != entityA && (this->collisionList.back())->getEntityB() != entityB ))
    c = CREngine::getInstance()->popCollisionObject();
  else
    c = this->collisionList.back();

  this->collisionList.push_back(c);

  // now register it as a shared collision with the other collision entity
  CollisionHandle* ch = entityB->getCollisionHandle(this->type);
  if( ch != NULL)
    ch->registerSharedCollision(c);

  return c;
}


/**
 * register a Collision to the Collision handle.
 *  @param collision
 * This is used for internal collision registration: sharing the collision objects between Collision Reactions
 * Therefore dispatching it only once
 */
void CollisionHandle::registerSharedCollision(Collision* collision)
{
  // set the state to not dispatched
  this->bDispatched = false;
  collision->setEntityBCollide(true);

  this->collisionList.push_back(collision);
}


/**
 * this is the function to be called on a collision event for this handle
 *  @param collision the collision objects containing all collision informations
 */
void CollisionHandle::registerCollisionEvent(CollisionEvent* collisionEvent)
{
  if( !this->filterCollisionEvent(collisionEvent))
    return;

  // set the state to not dispatched
  this->bDispatched = false;

  // checks if these WorldEntities have already collided or if its a new collision -> create a new Collision object
 Collision* c = this->registerCollision(collisionEvent->getEntityA(), collisionEvent->getEntityB());
 c->setEntityACollide(true);

 c->registerCollisionEvent(collisionEvent);
}


/**
 * flushes the collision list
 */
void CollisionHandle::flushCollisions()
{
  this->collisionList.clear();
}


/**
 * handles the collisions and react according to algorithm
 */
void CollisionHandle::handleCollisions()
{
  // collision reaction calculations (for every collision there will be a reaction)
  vector<Collision*>::iterator it = this->collisionList.begin();
  for(; it < this->collisionList.end(); it++)
  {
    (*it)->handleCollisionEvents();
  }

  // now set state to dispatched
  this->bDispatched = true;
  this->bCollided = false;
  this->flushCollisions();
}


/**
 * filter out the CollisionEvents that are not wanted
 *  @param collisionEvent the collision event to filter
 */
bool CollisionHandle::filterCollisionEvent(CollisionEvent* collisionEvent)
{
  vector<long>::iterator it = this->targetList.begin();
  for(; it < this->targetList.end(); it++)
    if( collisionEvent->getEntityA()->isA((ClassID)(*it)))
      return true;

  return false;
}








