

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "world_entity.h"
#include "shell_command.h"

#include "model.h"
#include "md2Model.h"
#include "util/loading/resource_manager.h"
#include "util/loading/load_param.h"
#include "vector.h"
#include "obb_tree.h"

#include "glgui_bar.h"

#include "state.h"
#include "camera.h"

#include "cr_engine.h"
#include "collision_handle.h"


using namespace std;

SHELL_COMMAND(model, WorldEntity, loadModel)
->describe("sets the Model of the WorldEntity")
->defaultValues("models/ships/fighter.obj", 1.0f);

SHELL_COMMAND(debugEntity, WorldEntity, debugWE);

/**
 *  Loads the WordEntity-specific Part of any derived Class
 *
 * @param root: Normally NULL, as the Derived Entities define a loadParams Function themeselves,
 *              that can calls WorldEntities loadParams for itself.
 */
WorldEntity::WorldEntity()
    : Synchronizeable()
{
  this->setClassID(CL_WORLD_ENTITY, "WorldEntity");

  this->obbTree = NULL;
  this->healthWidget = NULL;
  this->healthMax = 1.0f;
  this->health = 1.0f;
  this->scaling = 1.0f;

  /* OSOLETE */
  this->bVisible = true;
  this->bCollide = true;

  this->objectListNumber = OM_INIT;
  this->objectListIterator = NULL;

  this->toList(OM_NULL);

  modelFileName_handle = registerVarId( new SynchronizeableString( &modelFileName, &modelFileName, "modelFileName" ) );
  scaling_handle = registerVarId( new SynchronizeableFloat( &scaling, &scaling, "scaling" ) );
}

/**
 *  standard destructor
*/
WorldEntity::~WorldEntity ()
{
  State::getObjectManager()->toList(this, OM_INIT);

  // Delete the model (unregister it with the ResourceManager)
  for (unsigned int i = 0; i < this->models.size(); i++)
    this->setModel(NULL, i);

  // Delete the obbTree
  if( this->obbTree != NULL)
    delete this->obbTree;

  if (this->healthWidget != NULL)
    delete this->healthWidget;
}

/**
 * loads the WorldEntity Specific Parameters.
 * @param root: the XML-Element to load the Data From
 */
void WorldEntity::loadParams(const TiXmlElement* root)
{
  // Do the PNode loading stuff
  PNode::loadParams(root);

  LoadParam(root, "md2texture", this, WorldEntity, loadMD2Texture)
  .describe("the fileName of the texture, that should be loaded onto this world-entity. (must be relative to the data-dir)")
  .defaultValues("");

  // Model Loading
  LoadParam(root, "model", this, WorldEntity, loadModel)
  .describe("the fileName of the model, that should be loaded onto this world-entity. (must be relative to the data-dir)")
  .defaultValues("", 1.0f, 0);

  LoadParam(root, "maxHealth", this, WorldEntity, setHealthMax)
  .describe("The Maximum health that can be loaded onto this entity")
  .defaultValues(1.0f);

  LoadParam(root, "health", this, WorldEntity, setHealth)
  .describe("The Health the WorldEntity has at this moment")
  .defaultValues(1.0f);
}


/**
 * loads a Model onto a WorldEntity
 * @param fileName the name of the model to load
 * @param scaling the Scaling of the model
 *
 * FIXME
 * @todo: separate the obb tree generation from the model
 */
void WorldEntity::loadModel(const std::string& fileName, float scaling, unsigned int modelNumber, unsigned int obbTreeDepth)
{
  this->modelLODName = fileName;
  this->scaling = scaling;

  std::string name = fileName;

  if (  name.find( ResourceManager::getInstance()->getDataDir() ) == 0 )
  {
    name.erase(ResourceManager::getInstance()->getDataDir().size());
  }

  this->modelFileName = name;

  if (!fileName.empty())
  {
    // search for the special character # in the LoadParam
    if (fileName.find('#') != std::string::npos)
    {
      PRINTF(4)("Found # in %s... searching for LOD's\n", fileName.c_str());
      std::string lodFile = fileName;
      unsigned int offset = lodFile.find('#');
      for (unsigned int i = 0; i < 3; i++)
      {
        lodFile[offset] = 48+(int)i;
        if (ResourceManager::isInDataDir(lodFile))
          this->loadModel(lodFile, scaling, i);
      }
      return;
    }
    if (this->scaling <= 0.0)
    {
      PRINTF(1)("YOU GAVE ME A CRAPY SCALE resetting to 1.0\n");
      this->scaling = 1.0;
    }
    if(fileName.find(".obj") != std::string::npos)
    {
      PRINTF(4)("fetching OBJ file: %s\n", fileName.c_str());
      BaseObject* loadedModel = ResourceManager::getInstance()->load(fileName, OBJ, RP_CAMPAIGN, this->scaling);
      if (loadedModel != NULL)
        this->setModel(dynamic_cast<Model*>(loadedModel), modelNumber);
      else
        PRINTF(1)("OBJ-File %s not found.\n", fileName.c_str());

      if( modelNumber == 0)
        this->buildObbTree(obbTreeDepth);
    }
    else if(fileName.find(".md2") != std::string::npos)
    {
      PRINTF(4)("fetching MD2 file: %s\n", fileName.c_str());
      Model* m = new MD2Model(fileName, this->md2TextureFileName, this->scaling);
      //this->setModel((Model*)ResourceManager::getInstance()->load(fileName, MD2, RP_CAMPAIGN), 0);
      this->setModel(m, 0);

      if( m != NULL)
        this->buildObbTree(obbTreeDepth);
    }
  }
  else
  {
    this->setModel(NULL);
  }
}

/**
 * sets a specific Model for the Object.
 * @param model The Model to set
 * @param modelNumber the n'th model in the List to get.
 */
void WorldEntity::setModel(Model* model, unsigned int modelNumber)
{
  if (this->models.size() <= modelNumber)
    this->models.resize(modelNumber+1, NULL);

  if (this->models[modelNumber] != NULL)
  {
    Resource* resource = ResourceManager::getInstance()->locateResourceByPointer(dynamic_cast<BaseObject*>(this->models[modelNumber]));
    if (resource != NULL)
      ResourceManager::getInstance()->unload(resource, RP_LEVEL);
    else
    {
      PRINTF(4)("Forcing model deletion\n");
      delete this->models[modelNumber];
    }
  }

  this->models[modelNumber] = model;


  //   if (this->model != NULL)
  //     this->buildObbTree(4);
}


/**
 * builds the obb-tree
 * @param depth the depth to calculate
 */
bool WorldEntity::buildObbTree(int depth)
{
  if (this->obbTree)
    delete this->obbTree;

  if (this->models[0] != NULL)
  {
    this->obbTree = new OBBTree(depth, models[0]->getModelInfo(), this);
    return true;
  }
  else
  {
    PRINTF(1)("could not create obb-tree, because no model was loaded yet\n");
    this->obbTree = NULL;
    return false;
  }
}


/**
 * subscribes this world entity to a collision reaction
 *  @param type the type of reaction to subscribe to
 *  @param nrOfTargets number of target filters
 *  @param ... the targets as classIDs
 */
void WorldEntity::subscribeReaction(CREngine::CRType type, int nrOfTargets, ...)
{}


/**
 * @brief moves this entity to the List OM_List
 * @param list the list to set this Entity to.
 *
 * this is the same as a call to State::getObjectManager()->toList(entity , list);
 * directly, but with an easier interface.
 *
 * @todo inline this (peut etre)
 */
void WorldEntity::toList(OM_LIST list)
{
  State::getObjectManager()->toList(this, list);
}

void WorldEntity::toReflectionList()
{
  State::getObjectManager()->toReflectionList( this );
}

void removeFromReflectionList()
{
/// TODO
///  State::getObject
}

/**
 * sets the character attributes of a worldentity
 * @param character attributes
 *
 * these attributes don't have to be set, only use them, if you need them
*/
//void WorldEntity::setCharacterAttributes(CharacterAttributes* charAttr)
//{}


/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void WorldEntity::collidesWith(WorldEntity* entity, const Vector& location)
{
  /**
   * THIS IS A DEFAULT COLLISION-Effect.
   * IF YOU WANT TO CREATE A SPECIFIC COLLISION ON EACH OBJECT
   * USE::
   * if (entity->isA(CL_WHAT_YOU_ARE_LOOKING_FOR)) { printf "dothings"; };
   *
   * You can always define a default Action.... don't be affraid just test it :)
   */
  //  PRINTF(3)("collision %s vs %s @ (%f,%f,%f)\n", this->getClassName(), entity->getClassName(), location.x, location.y, location.z);
}


/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void WorldEntity::collidesWithGround(const Vector& location)
{
  PRINTF(0)("BSP_GROUND: %s collides \n", this->getClassName() );
}

void WorldEntity::collidesWithGround(const Vector& feet, const Vector& ray_1, const Vector& ray_2)
{
  
  // PRINTF(0)("BSP_GROUND: Player collides \n", this->getClassName() );
  
  Vector v = this->getAbsDirX();
  v.x *= 10;
  v.y *= 10;
  v.z *= 10;
  Vector u = this->getAbsDirY();
  
  if(feet.x == (u.x+this->getAbsCoor().x) &&  feet.y == u.y +this->getAbsCoor().y && feet.z == this->getAbsCoor().z)
  {
    
  this->setAbsCoor(ray_2 - v);
  }
  else
  {
    if(ray_1.x == this->getAbsCoor().x + v.x && ray_1.y == this->getAbsCoor().y + v.y + 0.1 && ray_1.z ==this->getAbsCoor().z + v.z)
    {
      this->setAbsCoor(feet -u );  
    }
      
    this->setAbsCoor(ray_2 - v);  
   
  }
}

/**
 *  this is called immediately after the Entity has been constructed, initialized and then Spawned into the World
 *
 */
void WorldEntity::postSpawn ()
{}


/**
 *  this method is called by the world if the WorldEntity leaves the game
 */
void WorldEntity::leaveWorld ()
{}


/**
 * resets the WorldEntity to its initial values. eg. used for multiplayer games: respawning
 */
void WorldEntity::reset()
{}

/**
 *  this method is called every frame
 * @param time: the time in seconds that has passed since the last tick
 *
 * Handle all stuff that should update with time inside this method (movement, animation, etc.)
*/
void WorldEntity::tick(float time)
{}


/**
 *  the entity is drawn onto the screen with this function
 *
 * This is a central function of an entity: call it to let the entity painted to the screen.
 * Just override this function with whatever you want to be drawn.
*/
void WorldEntity::draw() const
{
  //PRINTF(0)("(%s::%s)\n", this->getClassName(), this->getName());
  //  assert(!unlikely(this->models.empty()));
  {
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    /* translate */
    glTranslatef (this->getAbsCoor ().x,
                  this->getAbsCoor ().y,
                  this->getAbsCoor ().z);
    Vector tmpRot = this->getAbsDir().getSpacialAxis();
    glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );


    // This Draws the LOD's
    float cameraDistance = State::getCamera()->distance(this);
    if (cameraDistance > 30 && this->models.size() >= 3 && this->models[2] != NULL)
    {
      this->models[2]->draw();
    }
    else if (cameraDistance > 10 && this->models.size() >= 2 && this->models[1] != NULL)
    {
      this->models[1]->draw();
    }
    else if (this->models.size() >= 1 && this->models[0] != NULL)
    {
      this->models[0]->draw();
    }
    glPopMatrix();
  }
}

/**
 * @param health the Health to add.
 * @returns the health left (this->healthMax - health+this->health)
 */
float WorldEntity::increaseHealth(float health)
{
  this->health += health;
  if (this->health > this->healthMax)
  {
    float retHealth = this->healthMax - this->health;
    this->health = this->healthMax;
    this->updateHealthWidget();
    return retHealth;
  }
  this->updateHealthWidget();
  return 0.0;
}

/**
 * @param health the Health to be removed
 * @returns 0.0 or the rest, that was not substracted (bellow 0.0)
 */
float WorldEntity::decreaseHealth(float health)
{
  this->health -= health;

  if (this->health < 0)
  {
    float retHealth = -this->health;
    this->health = 0.0f;
    this->updateHealthWidget();
    return retHealth;
  }
  this->updateHealthWidget();
  return 0.0;

}

/**
 * @param maxHealth the maximal health that can be loaded onto the entity.
 */
void WorldEntity::setHealthMax(float healthMax)
{
  this->healthMax = healthMax;
  if (this->health > this->healthMax)
  {
    PRINTF(3)("new maxHealth is bigger as the old health. Did you really intend to do this for (%s::%s)\n", this->getClassName(), this->getName());
    this->health = this->healthMax;
  }
  this->updateHealthWidget();
}

/**
 * @brief creates the HealthWidget
 *
 * since not all entities need an HealthWidget, it is only created on request.
 */
void WorldEntity::createHealthWidget()
{
  if (this->healthWidget == NULL)
  {
    this->healthWidget = new OrxGui::GLGuiBar();
    this->healthWidget->setSize2D(30,400);
    this->healthWidget->setAbsCoor2D(10,100);

    this->updateHealthWidget();
  }
  else
    PRINTF(3)("Allready created the HealthWidget for %s::%s\n", this->getClassName(), this->getName());
}

void WorldEntity::increaseHealthMax(float increaseHealth)
{
  this->healthMax += increaseHealth;
  this->updateHealthWidget();
}


OrxGui::GLGuiWidget* WorldEntity::getHealthWidget()
{
  this->createHealthWidget();
  return this->healthWidget;
}

/**
 * @param visibility shows or hides the health-bar
 * (creates the widget if needed)
 */
void WorldEntity::setHealthWidgetVisibilit(bool visibility)
{
  if (visibility)
  {
    if (this->healthWidget != NULL)
      this->healthWidget->show();
    else
    {
      this->createHealthWidget();
      this->updateHealthWidget();
      this->healthWidget->show();
    }
  }
  else if (this->healthWidget != NULL)
    this->healthWidget->hide();
}

/**
 * @brief updates the HealthWidget
 */
void WorldEntity::updateHealthWidget()
{
  if (this->healthWidget != NULL)
  {
    this->healthWidget->setMaximum(this->healthMax);
    this->healthWidget->setValue(this->health);
  }
}


/**
 * DEBUG-DRAW OF THE BV-Tree.
 * @param depth What depth to draw
 * @param drawMode the mode to draw this entity under
 */
void WorldEntity::drawBVTree(int depth, int drawMode) const
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );


  if (this->obbTree)
    this->obbTree->drawBV(depth, drawMode);


  glPopMatrix();
}


/**
 * Debug the WorldEntity
 */
void WorldEntity::debugEntity() const
{
  PRINT(0)("WorldEntity %s::%s  (DEBUG)\n", this->getClassName(), this->getName());
  this->debugNode();
  PRINT(0)("List: %s ; ModelCount %d - ", ObjectManager::OMListToString(this->objectListNumber) , this->models.size());
  for (unsigned int i = 0; i < this->models.size(); i++)
  {
    if (models[i] != NULL)
      PRINT(0)(" : %d:%s", i, this->models[i]->getName());
  }
  PRINT(0)("\n");

}


/**
 * handler for changes on registred vars
 * @param id id's which changed
 */
void WorldEntity::varChangeHandler( std::list< int > & id )
{
  if ( std::find( id.begin(), id.end(), modelFileName_handle ) != id.end() ||
       std::find( id.begin(), id.end(), scaling_handle ) != id.end()
     )
  {
    loadModel( modelFileName, scaling );
  }

  PNode::varChangeHandler( id );
}

