#include <cassert>
#include <string>
#include <iostream>

#include <map>
#include <list>

#include "luaincl.h"
#include "Script.h"
#include "LuaCallback.h"
#include "scriptable.h"

// ---------------------------------------------------------------------------
namespace OrxScript
{
  /**
   * @brief Constructor
   *
   */
  Scriptable::Scriptable ()
  {
  }

  /**
   * @brief Deconstructor
   *
   * The Deconstructor tells all the scripts, that it is part of, that it is deleted.
   *
   */
  Scriptable::~Scriptable (void)
  {
    std::list<Script>::iterator it;
    for(it = scriptList.begin();it != scriptList.end(); it++)
    {
      (*it).script->removeFromScript((*it).thisReference);
    }

  }


  //Method indexing check
  int Scriptable::methods (LuaVirtualMachine& virtualMachine)
  {
    LuaScript* script =(getScriptByVirtualMachine(virtualMachine))->script;
    if(script)
    {
      int lastMethod = getLastMethodIndexByPointer(script);
      if(lastMethod != -1)
        return lastMethod;
    }

    return -1;
  }


  /**
   * @brief  Tells the scriptable that it got added to a script
   * @param  toScript a pointer to the script the object got added
   * @param
   * @param  reference a reference to the scriptable in that partilular script
   *
   *  The scriptable will register all its functions to the table at "reference"
   *  with the script.
   *
   */

  bool Scriptable::scriptableAdded(LuaScript* toScript, int toScriptRef, int reference)
  {

    bool success = true ;
    if(!scriptIsInScriptList(toScript))// look if the scriptable isn't already added.
      if(toScript)
      {
        Script newScript;
        newScript.script = toScript;
        newScript.scriptReference = toScriptRef;
        newScript.thisReference = reference;
        newScript.lastMethodIndex = -1;
        newScript.methodBase = -1;

        scriptList.push_back(newScript);

        int methodIndex;
        Script* tmpScript = getScriptByPointer(toScript);

        //add all the functions to the script
        std::list<std::string>::const_iterator it;

        for(it = functionList.begin();it != functionList.end(); it++)
        {

          if(tmpScript)
          {
            methodIndex = toScript->addFunctionToScriptable(*it, tmpScript->thisReference, tmpScript->lastMethodIndex);


            if(newScript.methodBase = -1)
            {
              if(methodIndex != -1)
              {
                std::cout<<"methodIndex is "<<methodIndex<<std::endl;
                tmpScript->methodBase= methodIndex;
                tmpScript->lastMethodIndex= methodIndex;
                tmpScript->functionMap[newScript.methodBase] = *it;
              }
              else{success = false; break;}
            }

            else if(methodIndex != -1)
            {

              tmpScript->lastMethodIndex = methodIndex;
              tmpScript->functionMap[newScript.lastMethodIndex] = *it;
            }
            else{success= false;break;}

          }
        }



      }
      else
      {
        success = false;
      }

    return success;

  }



  /**
    * @brief  Register a function with the scriptable
    * @param  functionName function name
    *
    *
   */

  void Scriptable::registerFunction(std::string functionName)
  {
    //check whether the function is already registered
    std::list<std::string>::iterator it;
    for(it = functionList.begin();it != functionList.end(); it++)
    {
      if((*it).compare(functionName) == 0)
      {
        break;
      }

    }

    if(it == functionList.end()) // if the functoin wasn't found
      functionList.push_back(functionName);
  }


  /**
   * @brief  Get the function name of the function at index
   * @param  index
   *
   * @return the function name on success.
   *
   * This function is used in the ScriptCalling function
   */
  std::string Scriptable::getFunctionAtIndex(int index, LuaVirtualMachine& virtualMachine)
  {
    lua_State * luaState = (lua_State* ) virtualMachine;

    if(virtualMachine.isOk())
    {
      lua_getglobal (luaState, "this");

      if (lua_istable (luaState, 1))
      {
        // Found the "this" table. The object pointer is at the index 0
        lua_rawgeti (luaState, 1, 0);

        if (lua_islightuserdata (luaState, -1))
        {
          // Found the pointer, need to cast it
          LuaScript* thisPointer = (LuaScript *) lua_touserdata (luaState, -1);

          if(thisPointer != NULL)
          {
            Script* script = getScriptByPointer(thisPointer);
            if( script != NULL)
            {
              std::map<int, std::string>::const_iterator it = script->functionMap.find(index);
              if(it != script->functionMap.end())//if found
                return script->functionMap[index];
            }

          }
        }
      }
    }
    return "";
  }


  /**
   * @brief Gets the lastMethod index associated with the LuaScript
   * @param the luaScript
   *
   * @return  A lua reference to the last function of the Scriptable
   *
   *
   */

  int Scriptable::getLastMethodIndexByPointer(LuaScript* luaScript)
  {
    if( luaScript )
    {
      Script* script = (getScriptByPointer(luaScript));
      if(script)
      {
        return (script->lastMethodIndex);
      }
    }

    return -1;
  }


  /**
   * @brief Gets the functionMap Associated with a LuaScript
   * @param the LuaScript
   *
   * @return A pointer to the functionMap, NULL on failure
   *
   *
   */

  std::map<int, std::string>* Scriptable::getFunctionMapByPointer(LuaScript* scriptPointer)
  {
    bool notFound = true;
    if(scriptPointer)
    {
      std::list<Script>::iterator it = scriptList.begin();


      while(notFound && it !=scriptList.end() )
      {
        if((*it).script == scriptPointer)
        {
          notFound = false;
          return &(it->functionMap);
        }
        it++;
      }
    }

    if(notFound)
      return NULL;

  }


  /**
   * @brief Gets the internal representation of a LuaScript by its pointer.
   * @param script the LuaScript
   *
   * @return returns the script if it was found, NULL else.
   *
   *
   */

  Script* Scriptable::getScriptByPointer(LuaScript* script)
  {
    bool notFound = true;

    if(script)
    {
      std::list<Script>::iterator it = scriptList.begin();

      while(notFound && it != scriptList.end() )
      {
        if((*it).script == script)
        {
          notFound = false;
          return &(*it);
        }
        it++;
      }

    }
    if(notFound)
      return NULL;

  }


  /**
   * @brief Extracts the Script out of the virtual machine
   * @param virtualMachine the virtualMachine to search for the script
   *
   * @return The Script. If there was an error it returns NULL
   *
   *
   */
  Script* Scriptable::getScriptByVirtualMachine(LuaVirtualMachine& virtualMachine)
  {

    if(virtualMachine.isOk())
    {
      lua_State * luaState = (lua_State* ) virtualMachine;
      lua_getglobal (luaState, "this");

      if (lua_istable (luaState, 1))
      {
        // Found the "this" table. The object pointer is at the index 0
        lua_rawgeti (luaState, 1, 0);

        if (lua_islightuserdata (luaState, -1))
        {
          // Found the pointer, need to cast it
          LuaScript* thisPointer = (LuaScript *) lua_touserdata (luaState, -1);
          Script* script = getScriptByPointer(thisPointer);

          if(script)
            return script;
          else
            return NULL;
        }
      }
    }


  }


  /**
   * @brief Checks if "this" Scriptable is already registred with a LuaScript
   * @param script The LuaScript
   *
   * @return true when the Scriptable is alreads registered with that script, flase else.
   *
   *
   */
  bool Scriptable::scriptIsInScriptList(LuaScript* script)
  {
    bool notFound = true;

    if(script)
    {
      std::list<Script>::iterator it = scriptList.begin();


      while( notFound && it !=scriptList.end() )
      {
        if((*it).script == script)
        {
          notFound = false;
          break;
        }
        it++;
      }

    }
    return !notFound;

  }


  /**
   * @brief Removes a LuaScript from the scriptList
   * @param deleted LuaScript that is about to be deleted
   *
   * This function has to be called int the destructor of the LuaScript.
   *
   *
   */

  void Scriptable::scriptDeleted(LuaScript* deleted)
  {
    if(deleted)
    {
      Script* script = getScriptByPointer(deleted);
      if(script)
      {
        std::list<Script>::iterator it = scriptList.begin();


        while((*it).script != deleted && it != scriptList.end() )
        {
          it++;
        }

        if(it != scriptList.end())
        {
          scriptList.erase(it);
        }


      }
    }

  }



  char Scriptable::whatIsThis()
  {
    char result = 's';
    return result;
  }
}
