/*!
 * @file multi_type.h
 * @brief Definition of a MultiType, that is able to hold one Value of many types.
 */

#ifndef _MULTI_TYPE_H
#define _MULTI_TYPE_H

#include <string>

// FORWARD DECLARATION

//! An enumerator defining Types, that can be stored inside a MultiType.
typedef enum
{
  MT_NULL            = 0,                  //!< No Value at all.
  MT_BOOL            = 1,                  //!< A bool Value.
  MT_INT             = 2,                  //!< An int Value.
  MT_UINT            = 2,
  MT_LONG            = 2,
  MT_FLOAT           = 4,                  //!< A float Value.
  MT_CHAR            = 8,                  //!< A single char.
  MT_STRING          = 16,                 //!< An entire String.
  MT_EXT1            = 32,                 //!< An external Type.
  MT_EXT2            = 64,                 //!< An external Type.
} MT_Type;



//! A class that encapsulates multiple differen types.
/**
 * Only one Value can be Stored inside this Class, but it can have any type: @see MT_Type.
 */
class MultiType {
  public:
    MultiType(MT_Type type = MT_NULL);
    MultiType(bool value);
    MultiType(int value);
    MultiType(double value);
    MultiType(char value);
    MultiType(const std::string& value);
    MultiType(const MultiType& multiType);
    virtual ~MultiType();

    MultiType& operator=(const MultiType& mt);
    MultiType& operator=(bool value) { this->setBool(value); return *this; };
    MultiType& operator=(int value) { this->setInt(value); return *this; };
    MultiType& operator=(float value) { this->setFloat(value); return *this; };
    MultiType& operator=(char value) { this->setChar(value); return *this; };
    MultiType& operator=(const std::string& value) { this->setString(value); return *this; };

    bool operator==(const MultiType& mt) const;
    bool operator==(bool value) const { return (this->getBool() == value); };
    bool operator==(int value) const { return (this->getInt() == value); };
    bool operator==(float value) const { return (this->getFloat() ==  value); };
    bool operator==(char value) const { return (this->getChar() == value); };
    bool operator==(const std::string& value) const { return (this->getString() == value); };
    bool operator==(MT_Type type) const { return (this->type == type); }
    bool operator!=(MT_Type type) const { return (this->type != type); }

    void setType(MT_Type type);

    void setBool(bool value);
    void setInt(int value);
    void setFloat(float value);
    void setChar(char value);
    void setString(const std::string& value);

    // for your convenience.
    inline void setValue(bool value) { this->setBool(value); };
    inline void setValue(int value) { this->setInt(value); };
    inline void setValue(float value) { this->setFloat(value); };
    inline void setValue(char value) { this->setChar(value); };
    inline void setValue(const char* value) { this->setString(value); };
    inline void setValue(const std::string& value) { this->setString(value); };
    void setValueOf(const MultiType& mt);

    /** @returns the Type of the Value stored in this MultiType */
    inline MT_Type getType() const { return this->type; };


    /* RETRIEVING FUNCTIONS */
    bool getBool() const;
    int getInt() const;
    float getFloat() const;
    char getChar() const;
    const char* getCString();
    std::string getString() const;
    const std::string& getConstString() const;

    void reset();

    void debug() const;

    static const std::string& MultiTypeToString(MT_Type type);
    static MT_Type StringToMultiType(const std::string& type);

  private:
    //! A union, that combines types into as little memory as possible.
    union MultiTypeValue
    {
      bool                       Bool;              //!< If it is a BOOL
      int                        Int;               //!< If it is an INT
      float                      Float;             //!< If it is a FLOAT
      char                       Char;              //!< If it is a CHAR
    }                            value;             //!< The Value.
    std::string                  storedString;      //!< The Stored String.
    MT_Type                      type;              //!< The Type stored in this MultiType


    static std::string           constString;       //!< A String for returning Constant strings.

    static const std::string     typeNames[];       //!< List of TypeNames for conversion.
};

#endif /* _MULTI_TYPE_H */
