/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "signal_connector.h"

namespace OrxGui
{

  /**
   * @brief creates a clean SignalConnector
   */
  SignalConnector::SignalConnector( )
  {
    this->object = NULL;
    this->exec = NULL;
  }

  /**
   * @brief Creates a SignalConnector out of an ObjectPointer, and an Executor.
   * @param object the Object the Executor will apply to.
   * @param executor the Executor that will be executed.
   * @return a new SignalConnector.
   */
  SignalConnector::SignalConnector(BaseObject* object, const Executor* executor)
  {
    this->object = object;
    this->exec = executor;
  };

  /**
   * @brief Creates a SignalConnector as a copy of another one.
   * @param signalConnector The SignalConnector to copy.
   */
  SignalConnector::SignalConnector(const SignalConnector& signalConnector)
  {
    this->object = signalConnector.object;
    this->exec = (signalConnector.exec == NULL) ? NULL : signalConnector.exec->clone();
  }

  /**
   * @brief deletes a SignalConnector.
   *
   * frees the stored executor
   */
  SignalConnector::~SignalConnector()
  {
    delete exec;
  }

  /**
   * @brief assignes a SignalConnector to the current one
   * @param signalConnector the SignalConnector to assign to this one
   * @return A Reference to this SignalConnector.
   */
  SignalConnector& SignalConnector::operator=(const SignalConnector& signalConnector)
  {
    delete this->exec;
    this->object = signalConnector.object;
    this->exec = (signalConnector.exec == NULL) ? NULL : signalConnector.exec->clone();
  }


  /**
   * @brief compares two SignalConnectors.
   * @param signalConnector the SignalConnector to compare against this one.
   * @return true if the Connectors are the same.
   */
  bool SignalConnector::operator==(const SignalConnector& signalConnector) const
  {
    return (this->object == signalConnector.object /* && this->exec == signalConnector.exec */ );
  }


  /**
   * @brief Executes the SignalConnector.
   */
  void SignalConnector::operator()() const
  {
    if (this->isValid())
      (*this->exec)(this->object, 0, NULL);
  }

  /**
   * @brief Executes the SignalConnector.
   * @param value0 First Value.
   */
  void SignalConnector::operator()(const MultiType& value0) const
  {
    if (exec != NULL && object != NULL)
      (*this->exec)(this->object, 1, &value0);
  }

  /**
   * @brief Executes the SignalConnector.
   * @param value0 First Value
   * @param value1 Second Value
   */
  void SignalConnector::operator()(const MultiType& value0, const MultiType& value1) const
  {
    if (exec != NULL && object != NULL)
    {
      MultiType mt[] = { value0, value1 };
      (*this->exec)(this->object, 2, mt);
    }
  }

  /**
   * @brief Executes the SignalConnector.
   * @param value0 First Value
   * @param value1 Second Value
   * @param value2 Third Value
   */
  void SignalConnector::operator()(const MultiType& value0, const MultiType& value1, const MultiType& value2) const
  {
    if (exec != NULL && object != NULL)
    {
      MultiType mt[] = { value0, value1, value2 };
      (*this->exec)(this->object, 3, mt);
    }
  }

  /**
   * @brief Executes the SignalConnector.
   * @param value0 First Value
   * @param value1 Second Value
   * @param value2 Third Value
   * @param value3 Fourth Value
   */
  void SignalConnector::operator()(const MultiType& value0, const MultiType& value1, const MultiType& value2, const MultiType& value3) const
  {
    if (exec != NULL && object != NULL)
    {
      MultiType mt[] = { value0, value1, value2, value3 };
      (*this->exec)(this->object, 4, mt);
    }
  }

  /**
   * @brief Executes the SignalConnector.
   * @param value0 First Value
   * @param value1 Second Value
   * @param value2 Third Value
   * @param value3 Fourth Value
   * @param value3 Fifth Value
   */
  void SignalConnector::operator()(const MultiType& value0, const MultiType& value1, const MultiType& value2, const MultiType& value3, const MultiType& value4) const
  {
    if (exec != NULL && object != NULL)
    {
      MultiType mt[] = { value0, value1, value2, value3, value4 };
      (*this->exec)(this->object, 5, mt);
    }
  }
}
