/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer:
*/

/*!
 * @file model.h
 *  Definition of an abstract model.
 *  containing all needed for other models
 */

#ifndef _MODEL_H
#define _MODEL_H

#include "base_object.h"
#include "vector.h"


//! holds the information about a triangle
typedef struct
{
  unsigned int   indexToVertices[3];   //!< index to the vertices of the triangle
  unsigned int   indexToNormals[3];    //!< index to the normals of the triangle
  unsigned int   indexToTexCoor[3];    //!< index to the texture coordinates
} sTriangleExt;


//! Model Information definitions
typedef struct
{
  unsigned int     numVertices;          //!< number of Vertices in the Model
  unsigned int     numTriangles;         //!< number of triangles in the Model
  unsigned int     numNormals;           //!< how many Normals in the Model
  unsigned int     numTexCoor;           //!< how many Texture Coordinates in the Model

  const float*     pVertices;            //!< array of the Vertices
  sTriangleExt*    pTriangles;           //!< array of all triangles
  const float*     pNormals;             //!< array of the Normals
  const float*     pTexCoor;             //!< array of the Texture Coordinates

} modelInfo;


//! This class defines the basic components of a model
class Model : virtual public BaseObject {

  public:
    virtual ~Model();

    virtual void draw() const;

    inline const modelInfo* getModelInfo() const { return &this->pModelInfo; }

    /** @returns a Pointer to the Vertex-Array, if it was deleted it returns NULL */
    inline const float* getVertexArray() const { return this->pModelInfo.pVertices; };
    /** @returns the VertexCount of this Model */
    inline unsigned int getVertexCount() const { return this->pModelInfo.numVertices; };

    /** @returns a Pointer to the Normals-Array, if it was deleted it returns NULL */
    inline const float* getNormalsArray() const { return this->pModelInfo.pNormals; };
    /** @returns the NormalsCount of this Model */
    inline unsigned int getNormalsCount() const { return this->pModelInfo.numNormals; };

    /** @returns a Pointer to the TexCoord-Array, if it was deleted it returns NULL */
    inline const float* getTexCoordArray() const { return this->pModelInfo.pTexCoor; };
    /** @returns the TexCoord-Count of this Model */
    inline unsigned int getTexCoordCount() const { return this->pModelInfo.numTexCoor; };

    /** @returns the Array of triangles */
    inline sTriangleExt* getTriangles() const { return this->pModelInfo.pTriangles; };
    /** @returns the Count of Faces of this Model */
    inline unsigned int getTriangleCount() const { return this->pModelInfo.numTriangles; };

  protected:
    Model();


  protected:
    modelInfo      pModelInfo;      //!< Reference to the modelInfo
};

#endif /* _MODEL_H */
