/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Stefan Lienard
   co-programmer: ...
*/

#include "mapped_water.h"
#include "util/loading/load_param.h"
#include "util/loading/factory.h"
#include "util/loading/resource_manager.h"
#include "state.h"




CREATE_FACTORY(MappedWater, CL_MAPPED_WATER);


MappedWater::MappedWater(const TiXmlElement* root)
{
  this->setClassID(CL_MAPPED_WATER, "MappedWater");
  this->toList(OM_ENVIRON);

  if (root != NULL)
    this->loadParams(root);

  PRINTF(0)("MaxTextureUnits: %i\n", Material::getMaxTextureUnits());

  // TODO rename texture to reflection texture
  /// loads the textures
  // set up refleciton texture
  // FIXME mat.setDiffuseMap(this->texture, 0); doesnt work,
  mat.setDiffuseMap("pictures/dudvmap.bmp", GL_TEXTURE_2D, 0);
  // load refraction texture
  mat.setDiffuseMap("pictures/error_texture.png", GL_TEXTURE_2D, 1);
  // load normal map
  mat.setDiffuseMap("pictures/normalmap.bmp", GL_TEXTURE_2D, 2);
  // load dudv map
  mat.setDiffuseMap("pictures/dudvmap.bmp", GL_TEXTURE_2D, 3);
  // set up depth texture
  mat.setDiffuseMap("pictures/sky-replace.jpg", GL_TEXTURE_2D, 4);



  /// MAKE THE MAPPING TEXTURE.
  // THIS IS A HACK AND SHOULD BE IN TEXTURE SOMEWHERE
  // set the size of the refraction and reflection textures
  this->textureSize = 512;
  unsigned int channels = 32;
  GLenum type = GL_RGBA;
  unsigned int* pTextureReflection = new unsigned int [this->textureSize * this->textureSize * channels];
  memset(pTextureReflection, 0, this->textureSize * this->textureSize * channels * sizeof(unsigned int));
  unsigned int* pTextureRefraction = new unsigned int [this->textureSize * this->textureSize * channels];
  memset(pTextureRefraction, 0, this->textureSize * this->textureSize * channels * sizeof(unsigned int));
  // Register the texture with OpenGL and bind it to the texture ID
  mat.select();
  glBindTexture(GL_TEXTURE_2D, this->mat.getDiffuseTexture(0));

  // Create the texture and store it on the video card
  glTexImage2D(GL_TEXTURE_2D, 0, channels, this->textureSize, this->textureSize, 0, type, GL_UNSIGNED_INT, pTextureReflection);

  //gluBuild2DMipmaps(GL_TEXTURE_2D, channels, this->textureSize, this->textureSize, type,  GL_UNSIGNED_INT, pTexture);

  //the same for the refraction
  glBindTexture(GL_TEXTURE_2D, this->mat.getDiffuseTexture(1));
  glTexImage2D(GL_TEXTURE_2D, 0, channels, this->textureSize, this->textureSize, 0, type, GL_UNSIGNED_INT, pTextureRefraction);

  // Set the texture quality
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

  // Since we stored the texture space with OpenGL, we can delete the image data
  delete [] pTextureReflection;
  delete [] pTextureRefraction;


  /// initialization of the texture coords, speeds etc...
  this->move = 0.0f;
  this->g_WaterUV = 35.0f;
  this->kNormalMapScale = 0.25f;
  this->g_WaterFlow = 0.0015f;

  /// initialization of the shaders
  // load shader files
/*  shader = new Shader( ResourceManager::getInstance()->getDataDir() + "/shaders/mapped_water.vert", ResourceManager::getInstance()->getDataDir() +"/shaders/mapped_water.frag");

  this->shader->activateShader();
  // Set the variable "reflection" to correspond to the first texture unit
  Shader::Uniform(shader, "reflection").set(0);
  // Set the variable "refraction" to correspond to the second texture unit
  Shader::Uniform(shader, "refraction").set(1);
  // Set the variable "normalMap" to correspond to the third texture unit
  Shader::Uniform(shader, "normalMap").set(2);
  // Set the variable "dudvMap" to correspond to the fourth texture unit
  Shader::Uniform(shader, "dudvMap").set(3);
  // Set the variable "depthMap" to correspond to the fifth texture unit
  Shader::Uniform(shader, "depthMap").set(4);
  // Give the variable "waterColor" a blue color
  Shader::Uniform(shader, "waterColor").set(0.1f, 0.2f, 0.4f, 1.0f);
  // Give the variable "lightPos" our hard coded light position
  Shader::Uniform(shader, "lightPos").set(lightPos.x, lightPos.y, lightPos.z, 1.0f);
  // uniform for the camera position
  cam_uni = new Shader::Uniform(shader, "cameraPos");

  this->shader->deactivateShader();*/
}

MappedWater::~MappedWater()
{
  //delete shader;
  //delete cam_uni;
}

void MappedWater::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "waterHeight", this, MappedWater, setHeight);
  LoadParam(root, "lightPos", this, MappedWater, setLightPosition);
}


void MappedWater::draw() const
{
  glPushMatrix();
  glTranslatef(0,this->waterHeight,0);

  mat.unselect();
  mat.select();

  ///this->shader->activateShader();

  // reset the camera uniform to the current cam position
  Vector pos = State::getCameraNode()->getAbsCoor();
  ///cam_uni->set(pos.x, pos.y, pos.z, 1.0f);

  glBegin(GL_QUADS);
  // The back left vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, 0.0f, g_WaterUV);            // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, 0.0f, refrUV - move);        // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, 0.0f, normalUV + move2);     // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                       // DUDV map texture
  glMultiTexCoord2f(GL_TEXTURE4, 0, 0);                       // Depth texture
  glVertex3f(0.0f, waterHeight, 0.0f);

  // The front left vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);                  // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f - move);           // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, 0.0f, 0.0f + move2);          // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                        // DUDV map texture
  glMultiTexCoord2f(GL_TEXTURE4, 0, 0);                        // Depth texture
  glVertex3f(0.0f, waterHeight, 1000.0f);

  // The front right vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, g_WaterUV, 0.0f);             // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, refrUV, 0.0f - move);         // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, normalUV, 0.0f + move2);      // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                        // DUDV map texture
  glMultiTexCoord2f(GL_TEXTURE4, 0, 0);                        // Depth texture
  glVertex3f(1000.0f, waterHeight, 1000.0f);

  // The back right vertice for the water
  glMultiTexCoord2f(GL_TEXTURE0, g_WaterUV, g_WaterUV);        // Reflection texture
  glMultiTexCoord2f(GL_TEXTURE1, refrUV, refrUV - move);       // Refraction texture
  glMultiTexCoord2f(GL_TEXTURE2, normalUV, normalUV + move2);  // Normal map texture
  glMultiTexCoord2f(GL_TEXTURE3, 0, 0);                        // DUDV map texture
  glMultiTexCoord2f(GL_TEXTURE4, 0, 0);                        // Depth texture
  glVertex3f(1000.0f, waterHeight, 0.0f);
  glEnd();

  ///this->shader->deactivateShader();

  mat.unselect();

  glPopMatrix();
}

void MappedWater::tick(float dt)
{
  // makes the water flow
  this->move += this->g_WaterFlow;
  this->move2 = this->move * this->kNormalMapScale;
  this->refrUV = this->g_WaterUV;
  this->normalUV = this->g_WaterUV * this->kNormalMapScale;
}

void MappedWater::setHeight(float height)
{
  this->waterHeight = height;
}

void MappedWater::activateReflection()
{
  // save viewport matrix and change the viewport size
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0,0, textureSize, textureSize);

  glPushMatrix();

  // If our camera is above the water we will render the scene flipped upside down.
  // In order to line up the reflection nicely with the world we have to translate
  // the world to the position of our reflected surface, multiplied by two.
  glEnable(GL_CLIP_PLANE0);
  Vector pos = State::getCameraNode()->getAbsCoor();
  if(pos.y > waterHeight)
  {
    // Translate the world, then flip it upside down
    glTranslatef(0.0f, waterHeight*2.0f, 0.0f);
    glScalef(1.0, -1.0, 1.0);

    // Since the world is updside down we need to change the culling to FRONT
    glCullFace(GL_FRONT);

    // Set our plane equation and turn clipping on
    double plane[4] = {0.0, 1.0, 0.0, -waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
  else
  {
    // If the camera is below the water we don't want to flip the world,
    // but just render it clipped so only the top is drawn.
    double plane[4] = {0.0, 1.0, 0.0, waterHeight};
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
}


void MappedWater::deactivateReflection()
{
  glDisable(GL_CLIP_PLANE0);
  glCullFace(GL_BACK);

  mat.select();
  glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, textureSize, textureSize);

  glPopMatrix();
  glPopAttrib();
}

void MappedWater::activateRefraction()
{/*
  // To create the refraction and depth textures we do the same thing
  // we did for the reflection texture, except we don't need to turn
  // the world upside down.  We want to find the depth of the water,
  // not the depth of the sky and above water terrain.

  // save viewport matrix and change the viewport size
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(0,0, textureSize, textureSize);

  glPushMatrix();

  // If our camera is above the water we will render only the parts that
  // are under the water.  If the camera is below the water we render
  // only the stuff above the water.  Like the reflection texture, we
  // incorporate clipping planes to only render what we need.

  // If the camera is above water, render the data below the water
  glEnable(GL_CLIP_PLANE0);
  Vector pos = State::getCameraNode()->getAbsCoor();
  if(pos.y > waterHeight)
  {
    double plane[4] = {0.0, -1.0, 0.0, waterHeight}; 
    glClipPlane(GL_CLIP_PLANE0, plane);
  }
  // If the camera is below the water, only render the data above the water
  else
  {
    glCullFace(GL_FRONT);
    double plane[4] = {0.0, 1.0, 0.0, -waterHeight}; 
    glClipPlane(GL_CLIP_PLANE0, plane);
  }*/
}

void MappedWater::deactivateRefraction()
{/*
  glDisable(GL_CLIP_PLANE0);
  glCullFace(GL_BACK);

  mat.select();
  // Bind the current scene to our refraction texture
//  glBindTexture(GL_TEXTURE_2D, g_Texture[REFRACTION_ID]);
  glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, textureSize, textureSize);

  // Bind the current scene to our depth texture
//  glBindTexture(GL_TEXTURE_2D, g_Texture[DEPTH_ID]);
  glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, 0, 0, textureSize, textureSize, 0);

  glPopMatrix();
  glPopAttrib();
  */
}
