/*!
 * @file cr_engine.h
 * @brief The collision reaction engine, defining generic collision reactions to collision events
 *
 * some parts of this module are tuned for efficiency. They are probably not self-explenatory anymore :D
 *  - Collision/ CollisionEvent objects recycling: This class contains a class of precached objects of these types
 *      they are used for fast registration of collision events: These objects can be get by the interface functions and
 *      are returned after one cycle automaticly by reseting the cached lists to its initial state. So do not wonder :D
 */

#ifndef _CR_ENGINE_
#define _CR_ENGINE_

#include "base_object.h"

#include <vector>

// FORWARD DECLARATION
class CollisionHandle;
class Collision;
class CollisionEvent;
class WorldEntity;

//! A default singleton class.
class CREngine : public BaseObject
{

  public:
  typedef enum CRType {
    CR_PHYSICS_MOMENTUM   = 0,
    CR_PHYSICS_GROUND,
    CR_PHYSICS_GROUND_WALK,

    CR_OBJECT_DAMAGE,
    CR_OBJECT_PICKUP,

    CR_VERTEX_TRAFO,

    CR_SPECIAL_CALLBACK,

    CR_NUMBER
  };

  virtual ~CREngine(void);

  /** @returns a Pointer to the only object of this Class */
  inline static CREngine* getInstance() { if (!singletonRef) singletonRef = new CREngine();  return singletonRef; };

  void reset();


  CollisionHandle* subscribeReaction(WorldEntity* worldEntity, CRType type);

  bool unsubscribeReaction(CollisionHandle* collisionHandle);

  void handleCollisions();

  /** @returns an instance to a collision object. instead of creating new object this ones can be resycled */
  inline Collision* popCollisionObject() {
    if( !this->collisionsUnused.empty()) {
      this->collisionsUsed.push_back(this->collisionsUnused.back()); this->collisionsUnused.pop_back(); return this->collisionsUsed.back(); } else return NULL; }

  /** @return an instanco of a CollisionEvent object. instead of creating a new object this ones can be used and resycled */
  inline CollisionEvent* popCollisionEventObject() {
    if( !this->collisionEventsUnused.empty()) {
      this->collisionEventsUsed.push_back(this->collisionEventsUnused.back()); this->collisionEventsUnused.pop_back(); return this->collisionEventsUsed.back(); } else return NULL; }

  void debug();


private:
  CREngine();
  void init();

  void flushCollisions();


private:
  std::vector<CollisionHandle*>       collisionHandles;         //!< list with the collision handles

  std::vector<Collision*>             collisionsUsed;           //!< a list of used, cached collisions
  std::vector<Collision*>             collisionsUnused;         //!< a list of unused, cached collisions

  std::vector<CollisionEvent*>        collisionEventsUsed;      //!< a list of used, cached collision events
  std::vector<CollisionEvent*>        collisionEventsUnused;    //!< a list of unused, cached collision events

  static CREngine*                    singletonRef;             //!< the reference to the CREngine object (singleton)
};

#endif /* _CR_ENGINE_ */
