/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2006 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: David Hasenfratz
*/

#include "billboard.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "graphics_engine.h"
#include "material.h"
#include "glincl.h"
#include "state.h"


using namespace std;


CREATE_FACTORY(Billboard, CL_BILLBOARD);


/**
 * standart constructor
 */
Billboard::Billboard (const TiXmlElement* root)
{
  this->init();

  if( root)
    this->loadParams(root);
}


/**
 * destroys a Billboard
 */
Billboard::~Billboard ()
{
  if (this->material)
    delete this->material;
}


/**
 * initializes the Billboard
 */
void Billboard::init()
{
  this->setClassID(CL_BILLBOARD, "Billboard");
  this->setName("Billboard");

  this->material = new Material();
  this->setTexture("pictures/error_texture.png");
  this->setAbsCoor(0, 0, 0);
  this->setVisibiliy(true);
  this->setSize(10, 10);
}


/**
 *  load params
 * @param root TiXmlElement object
 */
void Billboard::loadParams(const TiXmlElement* root)
{
  LoadParam(root, "texture", this->material, Material, setDiffuseMap)
      .describe("the texture-file to load onto the Billboard");

  LoadParam(root, "size", this, Billboard, setSize)
      .describe("the size of the Billboard in Pixels");
}


/**
 * sets the size of the Billboard.
 * @param size the size in pixels
 */
void Billboard::setSize(float sizeX, float sizeY)
{
  this->sizeX = sizeX;
  this->sizeY = sizeY;
}


/**
 * sets the material to load
 * @param textureFile The texture-file to load
 */
void Billboard::setTexture(const std::string& textureFile)
{
  this->material->setDiffuseMap(textureFile);
}


/**
 * ticks the Billboard
 * @param dt the time to ticks
 */
void Billboard::tick(float dt)
{/*
  float z = 0.0f;
  glReadPixels ((int)this->getAbsCoor2D().x,
                 GraphicsEngine::getInstance()->getResolutionY()-(int)this->getAbsCoor2D().y-1,
                 1,
                 1,
                 GL_DEPTH_COMPONENT,
                 GL_FLOAT,
                 &z);

  GLdouble objX=.0, objY=.0, objZ=.0;
  gluUnProject(this->getAbsCoor2D().x,
               GraphicsEngine::getInstance()->getResolutionY()-this->getAbsCoor2D().y-1,
               .99,  // z
               GraphicsEngine::modMat,
               GraphicsEngine::projMat,
               GraphicsEngine::viewPort,
               &objX,
               &objY,
  &objZ );*/
}


/**
 * draws the billboard
 */
void Billboard::draw() const
{
  if( !this->isVisible())
    return;

  glPushMatrix();

  glTranslatef(this->getAbsCoor().x, this->getAbsCoor().y, this->getAbsCoor().z);
  this->material->select();

  glBegin(GL_QUADS);
  glTexCoord2f(1.0f, 1.0f); glVertex3f(-sizeX/2, -sizeY/2,  0.0f);
  glTexCoord2f(0.0f, 1.0f); glVertex3f( sizeX/2, -sizeY/2,  0.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f( sizeX/2,  sizeY/2,  0.0f);
  glTexCoord2f(1.0f, 0.0f); glVertex3f(-sizeX/2,  sizeY/2,  0.0f);
  glEnd();

  glPopMatrix();
}
