/*!
 * @file world_entity.h
 * Definition of the basic WorldEntity
 */

#ifndef _WORLD_ENTITY_H
#define _WORLD_ENTITY_H

#include "p_node.h"
#include "synchronizeable.h"
#include "model.h"

#include "cr_engine.h"
#include "object_manager.h"
#include "glincl.h"
#include <vector>




// FORWARD DECLARATION
namespace OrxSound { class SoundBuffer; class SoundSource; }
namespace OrxGui { class GLGuiWidget; class GLGuiBar; };

class BVTree;
class BoundingVolume;
class Model;
class CollisionHandle;
class Collision;


//class CharacterAttributes;


//! Basis-class all interactive stuff in the world is derived from
class WorldEntity : public PNode
{
public:
  WorldEntity();
  virtual ~WorldEntity ();

  virtual void loadParams(const TiXmlElement* root);

  void loadModel(const std::string& fileName, float scaling = 1.0f, unsigned int modelNumber = 0, unsigned int obbTreeDepth = 4);
  void setModel(Model* model, unsigned int modelNumber = 0);
  Model* getModel(unsigned int modelNumber = 0) const { return (this->models.size() > modelNumber)? this->models[modelNumber] : NULL; };

  inline void loadMD2Texture(const std::string& fileName) { this->md2TextureFileName = fileName; }

  /** @param visibility if the Entity should be visible (been draw) */
  void setVisibiliy (bool visibility) { this->bVisible = visibility; };
  /** @returns true if the entity is visible, false otherwise */
  inline bool isVisible() const { return this->bVisible; };

  virtual void reset();

  virtual void postSpawn ();
  virtual void leaveWorld ();
  virtual void destroy() {};

  virtual void tick (float time);
  virtual void draw () const;

  /* --- Collision Detection Block  --- */
  bool buildObbTree(int depth);
  virtual void collidesWith (WorldEntity* entity, const Vector& location);
  /** @returns a reference to the obb tree of this worldentity */
  inline BVTree* getOBBTree() const { return this->obbTree; };
  void drawBVTree(int depth, int drawMode) const;

  /* --- Collision Reaction Block --- */
  void subscribeReaction(CREngine::CRType type, int nrOfTargets, long target, ...);
  bool registerCollision(WorldEntity* entityA, WorldEntity* entityB, BoundingVolume* bvA, BoundingVolume* bvB);


  /* @returns the Count of Faces on this WorldEntity */
  //unsigned int getFaceCount () const { return (this->model != NULL)?this->model->getFaceCount():0; };
  //  void addAbility(Ability* ability);
  //  void removeAbility(Ability* ability);
  //  void setCharacterAttributes(CharacterAttributes* charAttr);
  //  CharacterAttributes* getCharacterAttributes();

  /* --- Object Manager Block --- */
  void toList(OM_LIST list);
  /** @returns a Reference to the objectListNumber to set. */
  OM_LIST& getOMListNumber() { return this->objectListNumber; }
  /** @returns a Reference to the Iterator */
  ObjectManager::EntityList::iterator& getEntityIterator() { return this->objectListIterator; }

  /* --- Network Block --- */
  int       writeState(const byte* data, int length, int sender);
  int       readState(byte* data, int maxLength );

  /* --- Character Attribute Block --- */
  /** @returns the Energy of the entity */
  float getHealth() const { return this->health; };
  /** @returns the Maximum energy this entity can be charged with */
  float getHealthMax() const { return this->healthMax; }
  float increaseHealth(float health);
  float decreaseHealth(float health);
  void increaseHealthMax(float increaseHealth);
  OrxGui::GLGuiWidget* getHealthWidget();
  bool hasHealthWidget() const { return this->healthWidget; };

  /* --- Misc Stuff Block --- */
  void debugWE() { this->debugEntity(); }
  ;  ///FIXME
  void debugEntity() const;


protected:
  void setHealth(float health) { this->health = health; this->updateHealthWidget();};
  void setHealthWidgetVisibilit(bool visibility);
  void setHealthMax(float healthMax);
  void createHealthWidget();

  //  CharacterAttributes*    charAttr;         //!< the character attributes of a world_entity
private:
  void updateHealthWidget();

private:
  /// TODO maybe we will move the following three entries and the corresponding functions to Playable AND NPC
  float                   health;             //!< The Energy of this Entity, if the Entity has any energy at all.
  float                   healthMax;          //!< The Maximal energy this entity can take.
  OrxGui::GLGuiBar*       healthWidget;       //!< The Slider (if wanted).

  std::vector<Model*>     models;             //!< The model that should be loaded for this entity.
  std::string             md2TextureFileName; //!< the file name of the md2 model texture, only if this
  std::string             modelLODName;       //!< the name of the model lod file
  BVTree*                 obbTree;            //!< this is the obb tree reference needed for collision detection

  bool                    bCollide;           //!< If it should be considered for the collisiontest.
  bool                    bVisible;           //!< If it should be visible.

  OM_LIST                           objectListNumber;             //!< The ObjectList from ObjectManager this Entity is in.
  ObjectManager::EntityList::iterator objectListIterator;         //!< The iterator position of this Entity in the given list of the ObjectManager.

  float                   scaling;                                //!< the scaling of the model
  CollisionHandle*        collisionHandles[CREngine::CR_NUMBER];  //!< the list of the collision reactions


};

#endif /* _WORLD_ENTITY_H */
