/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file vector2D.h
 * A basic 2D Vector math framework
 *
 * Contains classes to handle vectors, lines, rotations and planes
*/

#ifndef __VECTOR2D_H_
#define __VECTOR2D_H_

#include <cmath>
#include "compiler.h"

//! small and performant 2D vector
typedef float sVec2D[2];

//! 2D Vector2D
/**
 *       Class to handle 2D Vector2Ds
 */
class Vector2D {
 public:
  Vector2D (float x, float y) : x(x), y(y) {}  //!< assignment constructor
  Vector2D () : x(0), y(0) {}

  /** @param v: the Vecor to compare with this one @returns true, if the Vecors are the same, false otherwise */
  inline bool operator== (const Vector2D& v) const { return (this->x==v.x && this->y==v.y); };
  /** @param v: the Vector to negative-compare with this one @returns true if the two vectors are different */
  inline bool operator!= (const Vector2D& v) const { return (this->x!=v.x && this->y!=v.y); };
  /** @param index The index of the "array" @returns the x/y coordinate */
  inline float operator[] (float index) const { return ( index == 0)? this->x : this->y; }
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector2D operator+ (const Vector2D& v) const { return Vector2D(x + v.x, y + v.y); };
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector2D operator+ (const sVec2D& v) const { return Vector2D(x + v[0], y + v[1]); };
  /** @param v The vector to add  @returns the addition between two vectors (this += v) */
  inline const Vector2D& operator+= (const Vector2D& v) { this->x += v.x; this->y += v.y; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline const Vector2D& operator+= (const sVec2D& v) { this->x += v[0]; this->y += v[1]; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector2D operator- (const Vector2D& v) const { return Vector2D(x - v.x, y - v.y); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector2D operator- (const sVec2D& v) const { return Vector2D(x - v[0], y - v[1]); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector2D& operator-= (const Vector2D& v) { this->x -= v.x; this->y -= v.y; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector2D& operator-= (const sVec2D& v) { this->x -= v[0]; this->y -= v[1]; return *this; };
  /** @param v the second vector  @returns The dotProduct between two vector (this (dot) v) */
  inline float operator* (const Vector2D& v) const { return x * v.x + y * v.y; };
  /** @todo strange */
  inline const Vector2D& operator*= (const Vector2D& v) { this->x *= v.x; this->y *= v.y; return *this; };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this * f) */
  inline Vector2D operator* (float f) const { return Vector2D(x * f, y * f); };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this *= f) */
  inline const Vector2D& operator*= (float f) { this->x *= f; this->y *= f; return *this; };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this / f) */
  inline Vector2D operator/ (float f) const { return (unlikely(f == 0.0)) ? Vector2D(0,0):Vector2D(this->x / f, this->y / f); };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this /= f) */
  inline const Vector2D& operator/= (float f) {if (unlikely(f == 0.0)) {this->x=0;this->y=0;} else {this->x /= f; this->y /= f;} return *this; };
  /**  copy constructor @todo (i do not know it this is faster) @param v the vector to assign to this vector. @returns the vector v */
  inline const Vector2D& operator= (const Vector2D& v) { this->x = v.x; this->y = v.y; return *this; };
  /** copy constructor* @param v the sVec3D to assign to this vector. @returns the vector v */
  inline const Vector2D& operator= (const sVec2D& v) { this->x = v[0]; this->y = v[1]; }
  /** @param v: the other vector \return the dot product of the vectors */
  float dot (const Vector2D& v) const { return x*v.x+y*v.y; };
  /** @param v multipy each entry with each other @returns this reference */
  const Vector2D& internalMultipy(const Vector2D& v) { this->x *= v.x; this->y *= y; };
  /** scales the this vector with v* @param v the vector to scale this with */
  void scale(const Vector2D& v) {   x *= v.x;  y *= v.y; };
  /** @returns the length of the vector */
  inline float len() const { return sqrt (x*x+y*y); }
  /** normalizes the vector */
  inline void normalize() { float l = len(); if( unlikely(l == 0.0))return; this->x=this->x/l; this->y=this->y/l; };
  Vector2D getNormalized() const;
  Vector2D abs();

  /** @param v the Vector to slerp to @param val 0 = stay 1 = at v */
  void slerp(const Vector2D& v, float val) { *this += (*this - v) * val; }

  /** @returns a Vector of (0,0) */
  static const Vector2D& nullVector() { static Vector2D nullVector; return nullVector; };

  void debug() const;

 public:
  float    x;     //!< The x Coordinate of the Vector2D.
  float    y;     //!< The y Coordinate of the Vector2D.
};


/** an easy way to create a Random Vector2D @param sideLength the length of the Vector2D (x not sqrt(x^2...)) */
#define VECTOR2D_RAND(sideLength)  (Vector2D((float)rand()/RAND_MAX -.5, (float)rand()/RAND_MAX -.5) * sideLength)


#endif /* __VECTOR2D_H_ */

