/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file vector2D.h
 * A basic 2D Vector math framework
 *
 * Contains classes to handle vectors, lines, rotations and planes
*/

#ifndef __RECT2D_H_
#define __RECT2D_H_

#include "vector2D.h"

//! 2D Rectangle 2D
/**
 *       Class to handle 2-Dimensional Rectangles.
 */
class Rect2D
{
public:
  Rect2D() { };
  Rect2D(float x, float y, float width, float height);
  Rect2D(const Vector2D& topLeft, const Vector2D& bottomRight );
  Rect2D(const Rect2D& rect);

  Rect2D& operator=(const Rect2D& rect);
  bool operator==(const Rect2D& rect) const;
  bool operator!=(const Rect2D& rect) const;

  Rect2D operator+(const Rect2D& rect) const;
  Rect2D& operator+=(const Rect2D& rect);

  Rect2D operator-(const Rect2D& rect) const;
  Rect2D& operator-=(const Rect2D& rect);

  Rect2D operator&(const Rect2D& rect) const;
  Rect2D& operator&=(const Rect2D& rect);

  Rect2D operator*(const Vector2D& scaling) const;



  /** @returns the top of the Rectangle */
  inline float top() const { return _topLeft.y; };
  /** @returns the bottom of the Rectangle */
  inline float bottom() const { return _bottomRight.y; };
  /** @returns the left border of the Rectangle */
  inline float left() const { return _topLeft.x; };
  /** @returns the right border of the rectangle */
  inline float right() const { return _bottomRight.x; };
  /** @returns the Center of the Rectangle */
  inline Vector2D center() const { return (_topLeft+_bottomRight)/ 2.0; }
  /** @returns the width of the Rectangle */
  inline float width() const { return (right() - left()); };
  /** @returns the height of the rectangle */
  inline float height() const { return (bottom() - top()); };
  /** @returns the Size of the Rectangle */
  inline Vector2D size() const { return Vector2D(width(), height()); }
  /** @returns the area of the Rectangle */
  float area() const { return width()*height(); }

  /** @param top sets the top border of the Rectangle */
  inline void setTop(float top) { _topLeft.y = top; };
  /** @param bottom the bottom border of the Rectangle */
  inline void setBottom(float bottom) { _bottomRight.y = bottom; };
  /** @param left the left border of the Rectangle */
  inline void setLeft(float left) { _topLeft.x = left; };
  /** @param right the right border of the Rectangle */
  inline void setRight(float right) { _bottomRight.x = right; };
  /** @param topLeft the top left corner of the Rectangle */
  inline void setTopLeft(const Vector2D& topLeft) { _topLeft = topLeft; };
  /** @param x the left border of the Rectangle @param y the top border */
  inline void setTopLeft(float x, float y) { this->setTopLeft(Vector2D(x,y)); };
  /** @param bottomRight the lower right corner of the Rectangle */
  inline void setBottomRight(const Vector2D& bottomRight) { _bottomRight = bottomRight; };
  /** @param x the right border of the Rectangle @param y the bottom border */
  inline void setBottomRight(float x, float y) { this->setBottomRight(Vector2D(x,y)); };

  void setWidth(float width);
  void setHeight(float height);
  void setSize(float width, float height);
  void setSize(const Vector2D& size);
  void setCenter(const Vector2D& center);

  void scaleX(float x);
  void scaleY(float y);
  void scale(float x, float y);
  void scale(const Vector2D& v);
  void scaleCentered(float x, float y);
  void scaleCentered(const Vector2D& v);

  void moveX(float x);
  void moveY(float y);
  void move(float x, float y);
  void move(const Vector2D& v);

  void swapTopBottom();
  void swapLeftRight();
  void normalize();
  Rect2D normalized() const;


  Rect2D intersection(const Rect2D& intersector) const;
  bool intersects(const Rect2D& intersector) const;
  Rect2D unite(const Rect2D& rect);
  bool contains(const Rect2D& rect) const;
  bool contains(const Vector2D& point) const;

  const Rect2D& slerp(const Rect2D& rect, float value);

private:
  Vector2D _topLeft;
  Vector2D _bottomRight;
};


#endif /* __RECT2D_H_ */

