/*!
 * @file glgui_widget.h
 * The gl_widget of the openglGUI
 */

#ifndef _GLGUI_WIDGET_H
#define _GLGUI_WIDGET_H

#include "element_2d.h"
#include "rect2D.h"

#include "material.h"

#include "event.h"
#include "signal_connector.h"

#include "glincl.h"

#include <vector>

// FORWARD DECLARATION
class Material;

namespace OrxGui
{
  typedef enum
  {
    Signal_click     = 0,
    Signal_release,
    Signal_rollOn,
    Signal_rollOff,
    Signal_open,
    Signal_close,
    Signal_destroy,

    SignalCount,
  } SignalType;


  class GLGuiCursor;

  //! if the Element should be visible by default.
#define GLGUI_WIDGET_DEFAULT_VISIBLE       false

  //! This is widget part of the openglGUI class
  /**
   * A widget is the main class of all the elements of th GUI.
   */
  class GLGuiWidget : public Element2D
  {

  private:

  public:
    GLGuiWidget(GLGuiWidget* parent = NULL);
    virtual ~GLGuiWidget();

    void show();
    void hide();

    /// INTERCONNECTIVITY
    void connectSignal(SignalType signalType, BaseObject* obj, const Executor* signal);
    void disconnectSignal(SignalType signalType);


    /// FOCUS
    /** @brief gives focus to this widget */
    void giveFocus();
    void breakFocus();
    /** @returns true if the widget is focusable */
    bool focusable() const { return this->_focusable; };
    /** @param focusable sets if the Widget should be focusable */
    void setFocusable(bool focusable = true) { this->_focusable = focusable; };
    /** @returns true if the position is inside of the Widget. @param position the position to check */
    bool focusOverWidget(const Vector2D& position) const;
    /** @brief overloaded function, that returns true if the cursor is on top of the Widget */
    bool focusOverWidget(const OrxGui::GLGuiCursor* const cursor) const;

    /** @returns the currently focused Widget (NULL if none is selected) */
    static GLGuiWidget* focused() { return GLGuiWidget::_focused; };


    /// CLICK
    void click();
    void release();
    bool clickable() const { return this->_clickable; };
    void setClickable(bool clickable = true) { this->_clickable = clickable; };

    virtual void update() {};
    virtual void draw() const;


    /// MATERIAL (looks)
    Material& backMaterial() { return this->_backMat; };
    const Material& backMaterial() const { return this->_backMat; };
    Rect2D& backRect() { return this->_backRect; };
    const Rect2D& backRect() const { return this->_backRect; };

    Material& frontMaterial() { return this->_frontMat; };
    const Material& frontMaterial() const { return this->_frontMat; };
    Rect2D& frontRect() { return this->_frontRect; };
    const Rect2D& frontRect() const { return this->_frontRect; };

    inline void drawRect(const Rect2D& rect) {
      glBegin(GL_QUADS);
      glTexCoord2i(0,0); glVertex2d(rect.left(), rect.top());
      glTexCoord2i(0,1); glVertex2d(rect.left(), rect.bottom());
      glTexCoord2i(1,1); glVertex2d(rect.right(), rect.bottom());
      glTexCoord2i(1,0); glVertex2d(rect.right(), rect.top());
      glEnd();
    }

    /** @param the Event to process. @returns true if the Event has been consumed*/
    virtual bool processEvent(const Event& event) { };


    DeclareSignal(testSignal, ());

  protected:
        // if something was clickt on the GUI-widget.
    virtual void clicked();
    virtual void released();
    virtual void receivedFocus();
    virtual void removedFocus();

    virtual void destroyed();


    inline void startDraw() const { glPushMatrix(); glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0); };
    inline void endDraw() const { glPopMatrix(); };

  private:
    void init();


  private:
    /// LOOKS
    Material                       _backMat;
    Rect2D                         _backRect;

    Material                       _frontMat;
    Rect2D                         _frontRect;


    /// SIGNALS
    std::vector<SignalConnector>   widgetSignals;

    /// EVENTS
    bool                           _focusable;        //!< If this widget can receive focus.
    bool                           _clickable;        //!< if this widget can be clicked upon.

    bool                           _pushed;

    static GLGuiWidget*            _focused;
    static GLGuiWidget*            _inputGrabber;
  };
}
#endif /* _GLGUI_WIDGET_H */
