/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: Patrick Boenzli : Vector2D::scale()
                                    Vector2D::abs()

   Benjamin Grauer: port to Vector2D
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_MATH

#include "rect2D.h"
#ifdef DEBUG
  #include "debug.h"
#else
  #include <stdio.h>
  #define PRINT(x) printf
#endif

/**
 * @brief creates a new Rectangle with
 * @param x the left border
 * @param y the top border
 * @param width: the width
 * @param height the height
 */
Rect2D::Rect2D(float x, float y, float width, float height)
{
  this->setLeft(x), this->setTop(y);
  this->setSize(width, height);
}

/**
 * @brief creates a new Rectangle with
 * @param topLeft the top-left corner
 * @param bottomRight the lower-right corner
 */
Rect2D::Rect2D(const Vector2D& topLeft, const Vector2D& bottomRight )
{
  this->setTopLeft(topLeft);
  this->setBottomRight(bottomRight);
}

/**
 * @brief creates a new Rectangle with
 * @param rect the rectangle to copy.
 */
Rect2D::Rect2D(const Rect2D& rect)
{
  *this = rect;
}

/**
 * @brief copies the Values of rect to this rect
 * @param rect copy the values from.
 * @returns this reference.
 */
Rect2D& Rect2D::operator=(const Rect2D& rect)
{
  this->_topLeft = rect._topLeft;
  this->_bottomRight = rect._bottomRight;
  return *this;
}

bool Rect2D::operator==(const Rect2D& rect) const
{
  return (this->_topLeft == rect._topLeft &&
      this->_bottomRight == rect._bottomRight);
}

bool Rect2D::operator!=(const Rect2D& rect) const
{
  return (this->_topLeft != rect._topLeft ||
      this->_bottomRight != rect._bottomRight);
}


Rect2D Rect2D::operator+(const Rect2D& rect) const
  {
    return Rect2D(this->_topLeft + rect._topLeft,
                  this->_bottomRight + rect._bottomRight);
  }


Rect2D& Rect2D::operator+=(const Rect2D& rect)
{
  this->_topLeft += rect._topLeft;
  this->_bottomRight += rect._bottomRight;
  return *this;
}


Rect2D Rect2D::operator-(const Rect2D& rect) const
{
  return Rect2D(this->_topLeft - rect._topLeft,
                this->_bottomRight - rect._bottomRight);
}

Rect2D& Rect2D::operator-=(const Rect2D& rect)
{
  this->_topLeft -= rect._topLeft;
  this->_bottomRight -= rect._bottomRight;
  return *this;
}

Rect2D Rect2D::operator&(const Rect2D& rect) const
  {
    return this->intersection(rect);
  }

Rect2D& Rect2D::operator&=(const Rect2D& rect)
{
  *this = this->intersection(rect);
}

Rect2D Rect2D::operator*(const Vector2D& scaling) const
{
#warning implements this...
  //this->scale(scaling);
}


void Rect2D::setWidth(float width)
{
  this->_bottomRight.x = this->_topLeft.x + width;
}


void Rect2D::setHeight(float height)
{
  this->_bottomRight.y = this->_topLeft.y + height;
}

void Rect2D::setSize(float width, float height)
{
  this->_bottomRight = this->_topLeft + Vector2D(width, height);
}

void Rect2D::setSize(const Vector2D& size)
{
  this->_bottomRight = this->_topLeft + size;
}

void Rect2D::setCenter(const Vector2D& center)
{
  this->move(center - this->center());
}

void Rect2D::scaleX(float x)
{
  this->_bottomRight.x = this->_topLeft.x + this->width() * x;
}

void Rect2D::scaleY(float y)
{
  this->_bottomRight.y = this->_topLeft.y + this->height() * y;
}

void Rect2D::scale(float x, float y)
{
  this->scale(Vector2D(x,y));
}

void Rect2D::scale(const Vector2D& v)
{
  this->_bottomRight = this->_topLeft + this->size().internalMultipy(v);

}

void Rect2D::scaleCentered(float x, float y)
{
  this->scaleCentered(Vector2D(x, y));
}

void Rect2D::scaleCentered(const Vector2D& v)
{
#warning implement this
}

void Rect2D::moveX(float x)
{
  this->_topLeft.x += x;
  this->_bottomRight.x += x;
}

void Rect2D::moveY(float y)
{
  this->_topLeft.y += y;
  this->_bottomRight.y += y;
}
void Rect2D::move(const Vector2D& v)
{
  this->_topLeft += v;
  this->_bottomRight += v;
}

void Rect2D::swapTopBottom()
{
  float tmp = this->top();
  this->setTop(this->bottom());
  this->setBottom(tmp);
}

void Rect2D::swapLeftRight()
{
  float tmp = this->left();
  this->setLeft(this->right());
  this->setRight(tmp);
}


void Rect2D::normalize()
{
  if (this->top() > this->bottom())
    this->swapTopBottom();
  if (this->left() > this->right())
    this->swapLeftRight();
}

Rect2D Rect2D::normalized() const
{
  Rect2D norm(*this);
  norm.normalize();
  return norm;
}


Rect2D Rect2D::intersection(const Rect2D& intersector) const
{
#warning implement
}

bool Rect2D::intersects(const Rect2D& intersector) const
  {
#warning implement

  }
Rect2D Rect2D::unite(const Rect2D& rect)
  {
#warning implement
  }
bool Rect2D::contains(const Rect2D& rect) const
  {
    return (this->top() <= rect.top() &&
        this->bottom() >= rect.bottom() &&
        this->left() <= rect.left() &&
        this->right() >= rect.right());
  }
bool Rect2D::contains(const Vector2D& point) const
  {
    return (this->top() <= point.y &&
        this->bottom() >= point.y &&
        this->left() <= point.x &&
        this->right() >= point.x);
  }

const Rect2D& Rect2D::slerp(const Rect2D& rect, float value)
{
  this->_topLeft.slerp(rect._topLeft, value);
  this->_bottomRight.slerp(rect._bottomRight, value);

  return *this;
}

