/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD


#include "simple_game_menu.h"

#include "event_handler.h"

#include "state.h"
#include "class_list.h"

#include "util/loading/load_param.h"
#include "fast_factory.h"
#include "util/loading/factory.h"

#include "world_entity.h"
#include "elements/image_entity.h"
#include "terrain.h"
#include "camera.h"

#include "graphics_engine.h"
#include "object_manager.h"
#include "sound_engine.h"
#include "sound_source.h"

#include "cd_engine.h"

#include "glgui.h"

//! This creates a Factory to fabricate a SimpleGameMenu
CREATE_FACTORY(SimpleGameMenu, CL_SIMPLE_GAME_MENU);



SimpleGameMenu::SimpleGameMenu(const TiXmlElement* root)
  : GameWorld()
{
  this->setClassID(CL_SIMPLE_GAME_MENU, "SimpleGameMenu");
  this->setName("SimpleGameMenu uninitialized");

  this->dataTank = new SimpleGameMenuData();

  this->cameraVector = Vector(50.0, 0.0, 0.0);
  this->menuLayers.push_back(MenuLayer());
  this->menuLayers.push_back(MenuLayer());

  this->layerIndex = 0;
  this->menuSelectedIndex = 0;
  this->selectorSource = NULL;

  OrxGui::GLGuiPushButton* pb = new OrxGui::GLGuiPushButton("PUSH ME");
  pb->show();
  pb->setAbsCoor2D(50, 50);

  OrxGui::GLGuiPushButton* dnpb = new OrxGui::GLGuiPushButton("DO NOT PUSH ME");
  dnpb->show();
  dnpb->setAbsCoor2D(350, 50);

  OrxGui::GLGuiPushButton* rdnpb = new OrxGui::GLGuiPushButton("REALLY DO NOT PUSH ME!!");
  rdnpb->show();
  rdnpb->setAbsCoor2D(200, 180);
  rdnpb->connectSignal(OrxGui::Signal_release, this, createExecutor<SimpleGameMenu>(&SimpleGameMenu::quitMenu));


  OrxGui::GLGuiHandler::getInstance()->activateCursor();

  if (root != NULL)
    this->loadParams(root);

  State::setMenuID(this->getNextStoryID());
}


/**
 *  @brief remove the SimpleGameMenu from memory
 *
 *  delete everything explicitly, that isn't contained in the parenting tree!
 *  things contained in the tree are deleted automaticaly
 */
SimpleGameMenu::~SimpleGameMenu ()
{
  PRINTF(3)("SimpleGameMenu::~SimpleGameMenu() - deleting current world\n");

  if( this->dataTank)
    delete this->dataTank;
}


/**
 * @brief loads the parameters of a SimpleGameMenu from an XML-element
 * @param root the XML-element to load from
 */
void SimpleGameMenu::loadParams(const TiXmlElement* root)
{
  /* skip the GameWorld, since it does not define any useful loadParams for this class */
  //static_cast<GameWorld*>(this)->loadParams(root);
  GameWorld::loadParams(root);

  PRINTF(4)("Loaded SimpleGameMenu specific stuff\n");
}


/**
 * @brief this is executed just before load
 *
 * since the load function sometimes needs data, that has been initialized
 * before the load and after the proceeding storyentity has finished
 */
ErrorMessage SimpleGameMenu::init()
{
  /* call underlying init funciton */
  GameWorld::init();

  this->subscribeEvent(ES_MENU, SDLK_UP);
  this->subscribeEvent(ES_MENU, SDLK_DOWN);
  this->subscribeEvent(ES_MENU, SDLK_RETURN);
  this->subscribeEvent(ES_MENU, SDLK_SPACE);
  this->subscribeEvent(ES_MENU, SDLK_ESCAPE);

  this->dataTank->localCamera->setRelCoor(this->cameraVector);

  GraphicsEngine::getInstance()->displayFPS(false);

  this->layerIndex = 0;
  this->menuSelectedIndex = 0;
}


/**
 * @brief load the data
 */
ErrorMessage SimpleGameMenu::loadData()
{
  GameWorld::loadData();

  if (this->dataXML != NULL)
  {
    LoadParam(dataXML, "selector-sound", this, SimpleGameMenu, setSelectorSound);

    TiXmlElement* element = this->dataXML->FirstChildElement("Elements");


    if( element == NULL)
    {
      PRINTF(1)("SimpleGameMenu is missing 'Elements'\n");
    }
    else
    {
      element = element->FirstChildElement();
    // load Players/Objects/Whatever
      PRINTF(4)("Loading Elements\n");
      while( element != NULL)
      {
        BaseObject* created = Factory::fabricate(element);
        if( created != NULL )
        {
          PRINTF(4)("Created a %s::%s\n", created->getClassName(), created->getName());
          if (!created->isA(CL_ELEMENT_2D))
            PRINTF(2)("Error the Created Entity is not an Element2D but an %s::%s\n", created->getClassName(), created->getName());
        }
        element = element->NextSiblingElement();
      }
      PRINTF(4)("Done loading Elements\n");
    }
  }

  /* get the menu list */
  const std::list<BaseObject*>* imageEntityList = ClassList::getList(CL_IMAGE_ENTITY);
  std::list<BaseObject*>::const_iterator entity;
  for (entity = imageEntityList->begin(); entity != imageEntityList->end(); entity++)
  {

    if( !strcmp("Selector_Menu", (*entity)->getName()))
    {
      this->menuSelector = dynamic_cast<ImageEntity*>(*entity);
      this->menuSelector->setBindNode((const PNode*)NULL);
    }
  }

  imageEntityList = ClassList::getList(CL_TEXT_ELEMENT);
  for (entity = imageEntityList->begin(); entity != imageEntityList->end(); entity++)
  {
    if( !strcmp( "StartGame_Menu", (*entity)->getName()))
    {
      this->menuStartGame = dynamic_cast<TextElement*>(*entity);
      this->menuStartGame->setBindNode((const PNode*)NULL);
      this->menuStartGame->setRelCoor2D(State::getResX() / 2.0f,
                                        State::getResY() / 2.0f - 60.0f);
      this->menuLayers[0].menuList.push_back(dynamic_cast<TextElement*>(*entity));

    }
    else if( !strcmp( "Multiplayer_Menu", (*entity)->getName()))
    {
      this->menuStartMultiplayerGame = dynamic_cast<TextElement*>(*entity);
      this->menuStartMultiplayerGame->setBindNode((const PNode*)NULL);
      this->menuStartMultiplayerGame->setRelCoor2D(State::getResX() / 2.0f,
                                                   State::getResY() / 2.0f + ((this->menuLayers[0].menuList.size() -1 ) * 60.0f));
      this->menuLayers[0].menuList.push_back(dynamic_cast<TextElement*>(*entity));
    }
    else if( !strcmp( "Quit_Menu", (*entity)->getName()))
    {
      this->menuQuitGame = dynamic_cast<TextElement*>(*entity);
      this->menuQuitGame->setBindNode((const PNode*)NULL);
      this->menuQuitGame->setRelCoor2D(State::getResX() / 2.0f,
                                       State::getResY() / 2.0f + ((this->menuLayers[0].menuList.size() -1 )* 60.0f));
      this->menuLayers[0].menuList.push_back(dynamic_cast<TextElement*>(*entity));
    }
  }
  this->menuSelected->getNullElement()->update2D(0.1f);
  this->menuSelectedIndex = 0;
  this->menuSelected = this->menuLayers[0].menuList[this->menuSelectedIndex];
  this->sliderTo(this->menuSelected, 0.0f);


  // loading the storyentities submenu (singleplayer)
  const std::list<BaseObject*>* storyEntities = ClassList::getList(CL_STORY_ENTITY);
  std::list<BaseObject*>::const_iterator it;
  for( it = storyEntities->begin(); it != storyEntities->end(); it++)
  {
    StoryEntity* se = dynamic_cast<StoryEntity*>(*it);
    if( se->isContainedInMenu())
    {
      this->menuLayers[1].storyList.push_back(se);

      // generating menu item
      TextElement* te = new TextElement();
      te->setVisibility(false);
      te->setText(se->getName());
      te->setRelCoor2D(State::getResX() / 2.0f - 200.0f, State::getResY() / 2.0f + ((this->menuLayers[1].menuList.size() - 2.0f) * 60.0f));
      this->menuLayers[1].menuList.push_back(te);

      // generating screenshoot item
      ImageEntity* ie = new ImageEntity();
      ie->setVisibility(false);
      ie->setBindNode((const PNode*)NULL);
      ie->setTexture(se->getMenuScreenshoot());
      ie->setRelCoor2D(State::getResX() / 2.0f + 250.0f, State::getResY() / 2.0f);
      ie->setSize2D(140.0f, 105.0f);
      this->menuLayers[1].screenshootList.push_back(ie);
    }
  }
}

/**
 * @brief set the Sound to play when switching menu entry.
 * @param selectorSound the sound to load.
 */
void SimpleGameMenu::setSelectorSound(const std::string& selectorSound)
{
  this->selectorSource = OrxSound::SoundEngine::getInstance()->createSource(selectorSound, NULL);
}

ErrorMessage SimpleGameMenu::unloadData()
{
  this->unsubscribeEvents(ES_MENU);

  std::vector<MenuLayer>::iterator mit;
  for(mit = this->menuLayers.begin(); mit != this->menuLayers.end(); mit++)
  {
    while(!(*mit).menuList.empty())
    {
      delete (*mit).menuList.back();
      (*mit).menuList.pop_back();
    }

    (*mit).menuList.clear();
    (*mit).storyList.clear();
    (*mit).screenshootList.clear();
  }

  // delete the SoundSource.
  if (this->selectorSource != NULL)
    delete this->selectorSource;
  this->selectorSource = NULL;

  GameWorld::unloadData();
}


/**
 * @brief start the menu
 */
bool SimpleGameMenu::start()
{
  EventHandler::getInstance()->pushState(ES_MENU);

  /* now call the underlying*/
  GameWorld::start();
}



/**
 * stop the menu
 */
bool SimpleGameMenu::stop()
{
  EventHandler::getInstance()->popState();

  /* now call the underlying*/
  GameWorld::stop();
}


/**
 *  override the standard tick for more functionality
 */
void SimpleGameMenu::tick()
{
  GameWorld::tick();

  // Make the GLGui tick.
  OrxGui::GLGuiHandler::getInstance()->tick(this->dtS);

  this->animateScene(this->dtS);
}


/**
 * @brief no collision detection in the menu
 */
void SimpleGameMenu::collide()
{
//   this->dataTank->localCamera->
}


/**
 * @brief animate the scene
 */
void SimpleGameMenu::animateScene(float dt)
{
  Quaternion q(/*0.00005*/ dt * .1, Vector(0.0, 1.0, 0.0));
  this->cameraVector = q.apply(this->cameraVector);
  this->dataTank->localCamera->setRelCoor(this->cameraVector);
  this->dataTank->localCamera->getTarget()->setRelCoorSoft(0,0,0);
}

void SimpleGameMenu::quitMenu()
{
  this->setNextStoryID(WORLD_ID_GAMEEND);
  this->stop();
}


/**
 * @brief event dispatcher funciton
 * @param event the incoming event
 */
void SimpleGameMenu::process(const Event &event)
{
  /* ----------------- LAYER 1 ---------------*/
  if( this->layerIndex == 0)
  {
    if( event.type == SDLK_RETURN && event.bPressed == true)
    {
      if( this->menuSelected == this->menuQuitGame)
      {
        this->setNextStoryID(WORLD_ID_GAMEEND);
        this->stop();
      }
      if( this->menuSelected == this->menuStartGame)
      {
        // switch to first submenu
        if( this->menuLayers[1].menuList.size() == 0)
        {
          PRINTF(1)("Haven't got any StoryEntities to play!\n");
          return;
        }

        this->switchMenuLayer(this->layerIndex, 1);
      }
    }
    if( event.type == SDLK_ESCAPE && event.bPressed == true)
    {
      this->setNextStoryID(WORLD_ID_GAMEEND);
      this->stop();
    }
  }  /* ----------------- LAYER 2 ---------------*/
  else if( this->layerIndex == 1)
  {
    if( event.type == SDLK_RETURN && event.bPressed == true)
    {
      this->setNextStoryID( this->menuLayers[1].storyList[this->menuSelectedIndex]->getStoryID());
      this->stop();
    }
    if( event.type == SDLK_ESCAPE && event.bPressed == true)
    {
      this->switchMenuLayer(this->layerIndex, 0);
    }
  }



  // The menu selction cursor
  if( event.type == SDLK_DOWN && event.bPressed == true)
  {
    if(this->menuSelectedIndex < (this->menuLayers[this->layerIndex].menuList.size() - 1))
    {
      this->menuSelected = this->menuLayers[this->layerIndex].menuList[++this->menuSelectedIndex];
      this->sliderTo(this->menuSelected, 5.0f);
      if (this->selectorSource != NULL)
        this->selectorSource->play();

      if( this->layerIndex == 1)
      {
        this->menuLayers[1].screenshootList[this->menuSelectedIndex]->setVisibility(true);
        this->menuLayers[1].screenshootList[this->menuSelectedIndex-1]->setVisibility(false);
      }
    }
  }
  else if( event.type == SDLK_UP && event.bPressed == true)
  {
    if(this->menuSelectedIndex > 0)
    {
      this->menuSelected = this->menuLayers[this->layerIndex].menuList[--this->menuSelectedIndex];
      this->sliderTo(this->menuSelected, 5.0f);
      if (this->selectorSource != NULL)
        this->selectorSource->play();

      if( this->layerIndex == 1)
      {
        this->menuLayers[1].screenshootList[this->menuSelectedIndex]->setVisibility(true);
        this->menuLayers[1].screenshootList[this->menuSelectedIndex+1]->setVisibility(false);
      }
    }
  }
}


/**
 * @brief switches to from one meny layer to an other
 * @param layer1 from layer
 * @param layer2 to layer
 */
void SimpleGameMenu::switchMenuLayer(int layer1, int layer2)
{
  // wrong sizes
  if(layer1 >= this->menuLayers.size() || layer2 >= this->menuLayers.size())
    return;


  PRINTF(0)("Removing layer %i\n", layer1);
  std::vector<TextElement*>::iterator te;
  // fade old menu
  for( te = this->menuLayers[layer1].menuList.begin(); te != this->menuLayers[layer1].menuList.end(); te++)
  {
    (*te)->setVisibility(false);
  }

  std::vector<ImageEntity*>::iterator it;

  //also fade the screenshots if in level choosement mode
  for( it = this->menuLayers[layer1].screenshootList.begin(); it != this->menuLayers[layer1].screenshootList.end(); it++)
  {
    (*it)->setVisibility(false);
  }


  PRINTF(0)("Showing layer %i\n", layer1);
  // beam here the new menu
  for( te = this->menuLayers[layer2].menuList.begin(); te != this->menuLayers[layer2].menuList.end(); te++ )
  {
    (*te)->setVisibility(true);
  }


  this->layerIndex = layer2;
  this->menuSelected = this->menuLayers[layer2].menuList[0];
  this->menuSelector->setAbsCoor2D(this->menuSelected->getAbsCoor2D() + Vector2D(0, this->menuSelected->getSizeY2D() *.5));
  this->menuSelector->setSize2D(this->menuSelected->getSizeX2D()*.7, this->menuSelected->getSizeY2D());
  this->menuSelectedIndex = 0;

  if( layer2 == 1)
    this->menuLayers[layer2].screenshootList[0]->setVisibility(true);
}

void SimpleGameMenu::sliderTo(const Element2D* element, float bias)
{
  if (bias > 0.0)
  {
    this->menuSelector->setAbsCoorSoft2D(element->getAbsCoor2D() + Vector2D(0, element->getSizeY2D() *.5), bias);
    this->menuSelector->setSizeSoft2D(element->getSizeX2D()*.7, element->getSizeY2D(), bias);
  }
  else
  {
    this->menuSelector->setAbsCoor2D(element->getAbsCoor2D() + Vector2D(0, element->getSizeY2D() *.5));
    this->menuSelector->setSize2D(element->getSizeX2D()*.7, element->getSizeY2D());
  }
}



/**********************************************************************************************
    SimpleGameMenuData
 **********************************************************************************************/


/**
 * SimpleGameMenuData constructor
 */
SimpleGameMenuData::SimpleGameMenuData()
{}

/**
 * SimpleGameMenuData decontructor
 */
SimpleGameMenuData::~SimpleGameMenuData()
{}


/**
 *  initialize the GameWorldDataData
 */
ErrorMessage SimpleGameMenuData::init()
{
  /* call underlying function */
  GameWorldData::init();
}


/**
 *  loads the GUI data
 * @param root reference to the xml root element
 */
ErrorMessage SimpleGameMenuData::loadGUI(const TiXmlElement* root)
{
  /* call underlying function */
  GameWorldData::loadGUI(root);
}


/**
 *  unloads the GUI data
 */
ErrorMessage SimpleGameMenuData::unloadGUI()
{
  /* call underlying function */
  GameWorldData::unloadGUI();
}


/**
 *  overloads the GameWorld::loadWorldEntities(...) class since the menu WorldEntity loading is different (less loading stuff)
 * @param root reference to the xml root parameter
 */
ErrorMessage SimpleGameMenuData::loadWorldEntities(const TiXmlElement* root)
{
  GameWorldData::loadWorldEntities(root);
  /*
  const TiXmlElement* element = root->FirstChildElement("WorldEntities");

  if( element != NULL)
  {
    element = element->FirstChildElement();
    PRINTF(4)("Loading WorldEntities\n");
    while(element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if( created != NULL )
        printf("Created a %s: %s\n", created->getClassName(), created->getName());

      if( element->Value() == "SkyBox")
        this->sky = dynamic_cast<WorldEntity*>(created);
      if( element->Value() == "Terrain")
        this->terrain = dynamic_cast<Terrain*>(created);
      element = element->NextSiblingElement();
    }

    PRINTF(4)("Done loading WorldEntities\n");
  }

  // init the pnode tree
  PNode::getNullParent()->init();
  */
}


/**
 *  unloads the world entities from the xml file
 */
ErrorMessage SimpleGameMenuData::unloadWorldEntities()
{
  /* call underlying function */
  GameWorldData::unloadWorldEntities();
}


/**
 *  loads the scene data
 * @param root reference to the xml root element
 */
ErrorMessage SimpleGameMenuData::loadScene(const TiXmlElement* root)
{
  /* call underlying function */
  GameWorldData::loadScene(root);
}


/**
 *  unloads the scene data
 */
ErrorMessage SimpleGameMenuData::unloadScene()
{
  /* call underlying function */
  GameWorldData::unloadScene();
}



