/*!
 * @file glgui_widget.h
 * The gl_widget of the openglGUI
 */

#ifndef _GLGUI_WIDGET_H
#define _GLGUI_WIDGET_H

#include "element_2d.h"
#include "event.h"
#include "material.h"

#include "glincl.h"
#include "signal_connector.h"

// FORWARD DECLARATION
class Material;

namespace OrxGui
{
  typedef enum
  {
    Signal_click     = 0,
    Signal_release,
    Signal_rollOn,
    Signal_rollOff,
    Signal_open,
    Signal_close,
    Signal_destroy,

    SignalCount,
  } SignalType;


  class GLGuiCursor;

  //! if the Element should be visible by default.
#define GLGUI_WIDGET_DEFAULT_VISIBLE       false

  //! This is widget part of the openglGUI class
  /**
   * A widget is the main class of all the elements of th GUI.
   */
  class GLGuiWidget : public Element2D
  {

  private:

  public:
    GLGuiWidget();
    virtual ~GLGuiWidget();

    void show();
    void hide();

    /// INTERCONNECTIVITY
    void connectSignal(SignalType signalType, BaseObject* obj, const Executor* signal);
    void disconnectSignal(SignalType signalType);


    /// FOCUS
    /** @brief gives focus to this widget */
    void giveFocus();
    void breakFocus();
    /** @returns true if the widget is focusable */
    bool focusable() const { return this->_focusable; };
    /** @param focusable sets if the Widget should be focusable */
    void setFocusable(bool focusable = true) { this->_focusable = focusable; };
    /** @returns true if the position is inside of the Widget. @param position the position to check */
    bool focusOverWidget(const Vector2D& position) const;
    /** @brief overloaded function, that returns true if the cursor is on top of the Widget */
    bool focusOverWidget(const OrxGui::GLGuiCursor* const cursor) const;

    /** @returns the currently focused Widget (NULL if none is selected) */
    static GLGuiWidget* focused() { return GLGuiWidget::_focused; };


    /// CLICK
    void click();
    void release();
    bool clickable() const { return this->_clickable; };
    void setClickable(bool clickable = true) { this->_clickable = clickable; };

    virtual void update() {};
    virtual void draw() const;


    /// MATERIAL (looks)
    Material& backMaterial() { return this->backMat; };
    const Material& backMaterial() const { return this->backMat; };

    Material& frontMaterial() { return this->frontMat; };
    const Material& frontMaterial() const { return this->frontMat; };





  protected:
        // if something was clickt on the GUI-widget.
    virtual void clicked() {};
    virtual void released() {};

    virtual void receivedFocus() {};
    virtual void removedFocus() {};

    virtual void destroyed() {};


    inline void startDraw() const { glPushMatrix(); glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0); };
    inline void endDraw() const { glPopMatrix(); };

  private:
    void init();


  protected:
    Material                       backMat;
    GLuint                         backModel;

    Material                       frontMat;
    GLuint                         frontModel;

  private:
    std::vector<SignalConnector>   widgetSignals;

    bool                           _focusable;        //!< If this widget can receive focus.
    bool                           _clickable;        //!< if this widget can be clicked upon.

    static GLGuiWidget*            _focused;
    static GLGuiWidget*            _inputGrabber;
  };
}
#endif /* _GLGUI_WIDGET_H */
