/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "element_2d.h"
#include "render_2d.h"

#include <algorithm>

// ONLY IF PNODE ENABLED //
#include "state.h"
#include "p_node.h"
#include "camera.h"
///////////////////////////

#include "graphics_engine.h"
#include "util/loading/load_param.h"
#include "class_list.h"

#include "color.h"

#include "shell_command.h"
SHELL_COMMAND(debug, Element2D, debug2DSC);


/**
 * @brief standard constructor
 * @param parent the parent to set for this Element2D
 *
 * NullElement2D needs this constructor with parameter NULL to initialize
 * itself. Otherwise it would result in an endless Loop.
 */
Element2D::Element2D (Element2D* parent, E2D_LAYER layer, short nodeFlags)
{
  this->setClassID(CL_ELEMENT_2D, "Element2D");

  this->setVisibility(true);
  this->bCurrentlyVisible = true;
  this->activate2D();
  this->setAlignment(E2D_ALIGN_NONE);
  this->bindNode = NULL;

  this->parentMode = nodeFlags;
  this->parent = NULL;
  this->absDirection = 0.0;
  this->relDirection = 0.0;
  this->bRelCoorChanged = true;
  this->bRelDirChanged = true;
  this->toCoordinate = NULL;
  this->toDirection = NULL;
  this->toSize = NULL;
  this->setSize2D(1, 1);


  this->layer = layer;
  if (parent != NULL)
    parent->addChild2D(this);
}


/**
 * @brief the mighty NullElement
 * TopMost Node of them all.
 */
Element2D* Element2D::nullElement = NULL;


/**
 * @brief standard deconstructor
 *
 * There are two general ways to delete an Element2D
 * 1. delete instance;
 *   -> result
 *    delete this Node and all its children and children's children...
 *    (danger if you still want the instance!!)
 *
 * 2. instance->remove2D(); delete instance;
 *   -> result:
 *    moves its children to the NullElement2D
 *    then deletes the Element.
 */
Element2D::~Element2D ()
{
  // remove the Element2D, delete it's children (if required).
  std::list<Element2D*>::iterator tmp = this->children.begin();
  std::list<Element2D*>::iterator deleteNode;
  while(!this->children.empty())
    while (tmp != this->children.end())
    {
      deleteNode = tmp;
      tmp++;
      //      printf("TEST::%s(%s) %s\n", (*deleteNode)->getName(), (*deleteNode)->getClassName(), this->getName());
      if ((this->parentMode & E2D_PROHIBIT_CHILD_DELETE) ||
          ((*deleteNode)->parentMode & E2D_PROHIBIT_DELETE_WITH_PARENT))
      {
        if (this == Element2D::nullElement && (*deleteNode)->parentMode & E2D_REPARENT_TO_NULL)
          delete (*deleteNode);
        else
          (*deleteNode)->reparent2D();
      }
      else
        delete (*deleteNode);
    }

  if (this->parent != NULL)
  {
    this->parent->eraseChild2D(this);
    this->parent = NULL;
  }

  // remove all other allocated memory.
  if (this->toCoordinate != NULL)
    delete this->toCoordinate;
  if (this->toDirection != NULL)
    delete this->toDirection;
  if (this->toSize != NULL)
    delete this->toSize;

  if (this == Element2D::nullElement)
    Element2D::nullElement = NULL;
}


/**
 * @brief Loads the Parameters of an Element2D from...
 * @param root The XML-element to load from
 */
void Element2D::loadParams(const TiXmlElement* root)
{
  BaseObject::loadParams(root);

  // ELEMENT2D-native settings.
  LoadParam(root, "alignment", this, Element2D, setAlignment)
  .describe("loads the alignment: (either: center, left, right or screen-center)");

  LoadParam(root, "layer", this, Element2D, setLayer)
  .describe("loads the layer onto which to project: (either: top, medium, bottom, below-all)");

  LoadParam(root, "bind-node", this, Element2D, setBindNode)
  .describe("sets a node, this 2D-Element should be shown upon (name of the node)");

  LoadParam(root, "visibility", this, Element2D, setVisibility)
  .describe("if the Element is visible or not");


  // PNode-style:
  LoadParam(root, "rel-coor-2d", this, Element2D, setRelCoor2D)
  .describe("Sets The relative position of the Node to its parent.");

  LoadParam(root, "abs-coor-2d", this, Element2D, setAbsCoor2D)
  .describe("Sets The absolute Position of the Node.");

  LoadParam(root, "rel-dir-2d", this, Element2D, setRelDir2D)
  .describe("Sets The relative rotation of the Node to its parent.");

  LoadParam(root, "abs-dir-2d", this, Element2D, setAbsDir2D)
  .describe("Sets The absolute rotation of the Node.");

  LoadParam(root, "parent", this, Element2D, setParent2D)
  .describe("the Name of the Parent of this Element2D");

  LoadParam(root, "parent-mode", this, Element2D, setParentMode2D)
  .describe("the mode to connect this node to its parent ()");

  // cycling properties
  LOAD_PARAM_START_CYCLE(root, element);
  {
    LoadParam_CYCLE(element, "child", this, Element2D, addChild2D)
    .describe("adds a new Child to the current Node.");
  }
  LOAD_PARAM_END_CYCLE(element);
}

/**
 * @brief sets the alignment of the 2D-element in form of a String
 * @param alignment the alignment @see loadParams
*/
void Element2D::setAlignment(const std::string& alignment)
{
  if (alignment == "center")
    this->setAlignment(E2D_ALIGN_CENTER);
  else if (alignment == "left")
    this->setAlignment(E2D_ALIGN_LEFT);
  else if (alignment == "right")
    this->setAlignment(E2D_ALIGN_RIGHT);
  else if (alignment == "screen-center")
    this->setAlignment(E2D_ALIGN_SCREEN_CENTER);
}


/**
 * @brief moves a Element to another layer
 * @param layer the Layer this is drawn on
 */
void Element2D::setLayer(E2D_LAYER layer)
{
  if (unlikely(this->layer == layer)) return;

  if (this->parent != NULL && this->parent->getLayer() > layer)
  {
    PRINTF(2)("Unable to set %s to layer %s, because it's parent(%s) is of higher layer %s\n",
              this->getName(),
              Element2D::layer2DToChar(layer),
              this->parent->getName(),
              Element2D::layer2DToChar(this->parent->getLayer()));
    layer = this->parent->getLayer();
  }
  this->layer = layer;


  if (this->parent != NULL)
    this->parent->children.sort(layerSortPredicate);
}

/**
 * @brief sets the layer onto which this 2D-element is projected to.
 * @param layer the layer @see loadParams @see Element2D::charToLayer2D(const std::string& layer)
 */
void Element2D::setLayer(const std::string& layer)
{
  this->setLayer(Element2D::charToLayer2D(layer));
}

void Element2D::setSizeSoft2D(float x, float y, float bias)
{
  if (likely(this->toSize == NULL))
    this->toSize = new Vector2D();

  *this->toSize = Vector2D(x,y);;
  this->bias = bias;
}


/**
 * @brief sets a node, this 2D-Element should be shown upon
 * @param bindNode the Node of the Node. (if NULL it will be unset).
 */
void Element2D::setBindNode(const PNode* bindNode)
{
  this->bindNode = bindNode;
  if (bindNode == NULL)
    this->bCurrentlyVisible = true;
}

/**
 * @brief sets a node, this 2D-Element should be shown upon
 * @param bindNode the name of the Node (should be existing)
 */
void Element2D::setBindNode(const std::string& bindNode)
{
  const PNode* tmpBindNode = dynamic_cast<const PNode*>(ClassList::getObject(bindNode, CL_PARENT_NODE));
  if (tmpBindNode != NULL)
    this->bindNode = tmpBindNode;
}

/**
 * @brief sets the relative coordinate of the Element2D to its parent
 * @param relCoord the relative coordinate to the parent
 */
void Element2D::setRelCoor2D (const Vector2D& relCoord)
{
  if (this->toCoordinate!= NULL)
  {
    delete this->toCoordinate;
    this->toCoordinate = NULL;
  }
  this->relCoordinate = relCoord;
  this->bRelCoorChanged = true;
}

/**
 * @brief sets the relative coordinate of the Element2D to its Parent
 * @param x the x coordinate
 * @param y the y coordinate
 */
void Element2D::setRelCoor2D (float x, float y)
{
  this->setRelCoor2D(Vector2D(x,y));
}

/**
 * @brief sets the Relative coordinate to the parent in Pixels
 * @param x the relCoord X
 * @param y the relCoord Y
 */
void Element2D::setRelCoor2Dpx (int x, int y)
{
  this->setRelCoor2D(Vector2D((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                              (float)y/(float)GraphicsEngine::getInstance()->getResolutionY()));
}

/**
 * @brief sets a new relative position smoothely
 * @param relCoordSoft the new Position to iterate to
 * @param bias how fast to iterate to this position
 */
void Element2D::setRelCoorSoft2D(const Vector2D& relCoordSoft, float bias)
{
  if (likely(this->toCoordinate == NULL))
    this->toCoordinate = new Vector2D();

  *this->toCoordinate = relCoordSoft;
  this->bias = bias;
}

/**
 * @brief sets a new relative position smoothely
 * @param x the new x-coordinate in Pixels of the Position to iterate to
 * @param y the new y-coordinate in Pixels of the Position to iterate to
 * @param bias how fast to iterate to this position
 */
void Element2D::setRelCoorSoft2Dpx (int x, int y, float bias)
{
  this->setRelCoorSoft2D(Vector2D((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                                  (float)y/(float)GraphicsEngine::getInstance()->getResolutionY()),
                         bias);
}

/**
 * @brief set relative coordinates smoothely
 * @param x x-relative coordinates to its parent
 * @param y y-relative coordinates to its parent
 * @param z z-relative coordinates to its parent
 * @see  void PNode::setRelCoorSoft (const Vector2D&, float)
 */
void Element2D::setRelCoorSoft2D(float x, float y, float bias)
{
  this->setRelCoorSoft2D(Vector2D(x, y), bias);
}

/**
 * @param absCoord set absolute coordinate
 */
void Element2D::setAbsCoor2D (const Vector2D& absCoord)
{
  if (this->toCoordinate!= NULL)
  {
    delete this->toCoordinate;
    this->toCoordinate = NULL;
  }

  if( likely(this->parentMode & E2D_PARENT_MOVEMENT))
  {
    /* if you have set the absolute coordinates this overrides all other changes */
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }
  if( this->parentMode & E2D_PARENT_ROTATE_MOVEMENT)
  {
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }

  this->bRelCoorChanged = true;
}

/**
 * @param x x-coordinate.
 * @param y y-coordinate.
 * @param z z-coordinate.
 * @see void PNode::setAbsCoor (const Vector2D& absCoord)
 */
void Element2D::setAbsCoor2D (float x, float y)
{
  this->setAbsCoor2D(Vector2D(x, y));
}

/**
 * @param x x-coordinate in Pixels
 * @param y y-coordinate in Pixels
 * @see void PNode::setAbsCoor (const Vector2D& absCoord)
 */
void Element2D::setAbsCoor2Dpx (int x, int y)
{
  this->setAbsCoor2D(Vector2D((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                              (float)y/(float)GraphicsEngine::getInstance()->getResolutionY()));
}

/**
 * @param absCoordSoft set absolute coordinate
 * @param bias how fast to iterato to the new Coordinate
 */
void Element2D::setAbsCoorSoft2D (const Vector2D& absCoordSoft, float bias)
{
  if (this->toCoordinate == NULL)
    this->toCoordinate = new Vector2D();

  if( likely(this->parentMode & E2D_PARENT_MOVEMENT))
  {
    /* if you have set the absolute coordinates this overrides all other changes */
    if (likely(this->parent != NULL))
      *this->toCoordinate = absCoordSoft - parent->getAbsCoor2D ();
    else
      *this->toCoordinate = absCoordSoft;
  }
  if( this->parentMode & E2D_PARENT_ROTATE_MOVEMENT)
  {
    if (likely(this->parent != NULL))
      *this->toCoordinate = absCoordSoft - parent->getAbsCoor2D ();
    else
      *this->toCoordinate = absCoordSoft;
  }

  this->bias = bias;
}

/**
 * @param x x-coordinate.
 * @param y y-coordinate.
 * @param z z-coordinate.
 * @see void PNode::setAbsCoor (const Vector2D& absCoord)
 */
void Element2D::setAbsCoorSoft2D (float x, float y, float bias)
{
  this->setAbsCoorSoft2D(Vector2D(x, y), bias);
}

/**
 * @brief shift coordinate ralative
 * @param shift shift vector
 *
 * This simply adds the shift-Vector2D to the relative Coordinate
 */
void Element2D::shiftCoor2D (const Vector2D& shift)
{
  this->relCoordinate += shift;
  this->bRelCoorChanged = true;

}

/**
 * @brief shifts in PixelSpace
 * @param x the pixels to shift in X
 * @param y the pixels to shift in Y
 */
void Element2D::shiftCoor2Dpx (int x, int y)
{
  this->shiftCoor2D(Vector2D((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                             (float)y/(float)GraphicsEngine::getInstance()->getResolutionY()));
}

/**
 * @brief set relative direction
 * @param relDir to its parent
 */
void Element2D::setRelDir2D (float relDir)
{
  if (this->toDirection!= NULL)
  {
    delete this->toDirection;
    this->toDirection = NULL;
  }

  this->relDirection = relDir;
  this->bRelDirChanged = true;
}

/**
 * @brief sets the Relative Direction of this node to its parent in a Smoothed way
 * @param relDirSoft the direction to iterate to smoothely.
 * @param bias how fast to iterate to the new Direction
 */
void Element2D::setRelDirSoft2D(float relDirSoft, float bias)
{
  if (likely(this->toDirection == NULL))
    this->toDirection = new float;

  *this->toDirection = relDirSoft;
  this->bias = bias;
}

/**
 * @brief sets the absolute direction
 * @param absDir absolute coordinates
 */
void Element2D::setAbsDir2D (float absDir)
{
  if (this->toDirection!= NULL)
  {
    delete this->toDirection;
    this->toDirection = NULL;
  }

  if (likely(this->parent != NULL))
    this->relDirection = absDir - this->parent->getAbsDir2D();
  else
    this->relDirection = absDir;

  this->bRelDirChanged = true;
}

/**
 * @brief sets the absolute direction softly
 * @param absDir absolute coordinates
 */
void Element2D::setAbsDirSoft2D (float absDirSoft, float bias)
{
  if (this->toDirection == NULL)
    this->toDirection = new float;

  if (likely(this->parent != NULL))
    *this->toDirection = absDirSoft - this->parent->getAbsDir2D();
  else
    *this->toDirection = absDirSoft;

  this->bias = bias;
}

/**
 * shift Direction
 * @param shift the direction around which to shift.
 */
void Element2D::shiftDir2D (float shiftDir)
{
  this->relDirection = this->relDirection + shiftDir;
  this->bRelDirChanged = true;
}

/**
 * @brief adds a child and makes this node to a parent
 * @param child child reference
 * @param parentMode on which changes the child should also change ist state
 *
 * use this to add a child to this node.
 */
void Element2D::addChild2D (Element2D* child)
{
  assert(child != NULL);
  if( likely(child->parent != NULL))
  {
    PRINTF(5)("Element2D::addChild() - reparenting node: removing it and adding it again\n");
    child->parent->eraseChild2D(child);
  }
  if (this->checkIntegrity(child))
  {
    // Setting the New Parent.
    child->parent = this;
    if (likely(this != NULL))
    {
      // Layers of Children that are smaller than this(parents) Layer will be updated, and pushed to the front.
      if (unlikely(this->layer > child->getLayer()))
      {
        PRINTF(2)("Layer '%s' of Child(%s::%s) lower than parents(%s::%s) layer '%s'. updating...\n",
                  Element2D::layer2DToChar(child->getLayer()),child->getClassName(), child->getName(),
                  this->getClassName(), this->getName(), Element2D::layer2DToChar(this->layer));
        child->layer = this->layer;
        this->children.push_front(child);
      }
      else
      {
        // Inserting the Element at the right Layer depth.
        std::list<Element2D*>::iterator elem;
        for (elem = this->children.begin(); elem != this->children.end(); elem++)
        {
          if ((*elem)->layer <= child->layer)
          {
            this->children.insert(elem, child);
            break;
          }
        }
        // if we are at the Last child push it back.
        if (elem == this->children.end())
          this->children.push_back(child);
      }
    }
    else
    {
      PRINTF(1)("Tried to reparent2D to own child '%s::%s' to '%s::%s'.\n",
                this->getClassName(), this->getName(), child->getClassName(), child->getName());
      child->parent = NULL;
    }
  }
  child->parentCoorChanged2D();
}

/**
 * @see Element2D::addChild(Element2D* child);
 * @param childName the name of the child to add to this PNode
 */
void Element2D::addChild2D (const std::string& childName)
{
  Element2D* childNode = dynamic_cast<Element2D*>(ClassList::getObject(childName, CL_ELEMENT_2D));
  if (childNode != NULL)
    this->addChild2D(childNode);
}

/**
 * @brief removes a child from the node
 * @param child the child to remove from this Node..
 *
 * Children from nodes will not be lost, they are referenced to NullPointer
 */
void Element2D::removeChild2D (Element2D* child)
{
  if (child != NULL)
    child->remove2D();
}

/**
 * !! PRIVATE FUNCTION
 * @brief reparents an Element2D (happens on Parents Node delete or remove and Flags are set.)
 */
void Element2D::reparent2D()
{
  if (this->parentMode & E2D_REPARENT_TO_NULL)
    this->setParent2D((Element2D*)NULL);
  else if (this->parentMode & E2D_REPARENT_TO_PARENTS_PARENT && this->parent != NULL)
    this->setParent2D(this->parent->getParent2D());
  else
    this->setParent2D(Element2D::getNullElement());
}


/**
 * @param child the child to be erased from this Nodes List
 */
void Element2D::eraseChild2D(Element2D* child)
{
  assert (this != NULL && child != NULL);
  std::list<Element2D*>::iterator childIT = std::find(this->children.begin(), this->children.end(), child);
  this->children.erase(childIT);
}



/**
 * @brief remove this Element from the tree and adds all children to NullElement2D
 *
 * afterwards this Node is free, and can be reattached, or deleted freely.
 */
void Element2D::remove2D()
{
  std::list<Element2D*>::iterator child = this->children.begin();
  std::list<Element2D*>::iterator reparenter;
  while (child != this->children.end())
  {
    reparenter = child;
    child++;
    if (this->parentMode & E2D_REPARENT_CHILDREN_ON_REMOVE ||
        (*reparenter)->parentMode & E2D_REPARENT_ON_PARENTS_REMOVE)
    {
      printf("TEST----------------%s ---- %s\n", this->getClassName(), (*reparenter)->getClassName());
      (*reparenter)->reparent2D();
      printf("REPARENTED TO: %s::%s\n",(*reparenter)->getParent2D()->getClassName(),(*reparenter)->getParent2D()->getName());
    }
  }
  if (this->parent != NULL)
  {
    this->parent->eraseChild2D(this);
    this->parent = NULL;
  }
}


/**
 * @see Element2D::setParent(Element2D* parent);
 * @param parentName the name of the Parent to set to this Element2D
 */
void Element2D::setParent2D (const std::string& parentName)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    parentNode->addChild2D(this);
  else
    PRINTF(2)("Not Found Element2D's (%s::%s) new Parent by Name: %s\n",
              this->getClassName(), this->getName(), parentName.c_str());
}

/**
 * @brief does the reparenting in a very smooth way
 * @param parentNode the new Node to connect this node to.
 * @param bias the speed to iterate to this new Positions
 */
void Element2D::setParentSoft2D(Element2D* parentNode, float bias)
{
  if (this->parent == parentNode)
    return;

  if (likely(this->toCoordinate == NULL))
  {
    this->toCoordinate = new Vector2D();
    *this->toCoordinate = this->getRelCoor2D();
  }
  if (likely(this->toDirection == NULL))
  {
    this->toDirection = new float;
    *this->toDirection = this->getRelDir2D();
  }
  this->bias = bias;


  Vector2D tmpV = this->getAbsCoor2D();
  float tmpQ = this->getAbsDir2D();

  parentNode->addChild2D(this);

  if (this->parentMode & E2D_PARENT_ROTATE_MOVEMENT) //! @todo implement this.
    ;//this->setRelCoor(this->parent->getAbsDir().inverse().apply(tmpV - this->parent->getAbsCoor()));
  else
    this->relCoordinate = (tmpV - parentNode->getAbsCoor2D());
  this->bRelCoorChanged = true;

  this->relDirection = (tmpQ - parentNode->getAbsDir2D());
  this->bRelDirChanged = true;
}

/**
 * @brief does the reparenting in a very smooth way
 * @param parentName the name of the Parent to reconnect to
 * @param bias the speed to iterate to this new Positions
 */
void Element2D::setParentSoft2D(const std::string& parentName, float bias)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    this->setParentSoft2D(parentNode, bias);
}

/**
 * @param parentMode sets the parentingMode of this Node
 */
void Element2D::setParentMode2D(E2D_PARENT_MODE parentMode)
{
  this->parentMode = ((this->parentMode & 0xfff0) | parentMode);
}


/**
 * @brief sets the mode of this parent manually
 * @param parentMode a String representing this parentingMode
 */
void Element2D::setParentMode2D (const std::string& parentingMode)
{
  this->setParentMode2D(Element2D::stringToParentingMode2D(parentingMode));
}

/**
 * @brief checks if elem1 is in a deeper layer as elem2
 * @param elem1 the first Element2D
 * @param elem2 the second Element2D
 * @returns true if elem1->layer < elem2->layer
 */
bool Element2D::layerSortPredicate(const Element2D* elem1, const Element2D* elem2)
{
  return elem1->layer < elem2->layer;
}


/**
 * @returns the NullElement (and if needed (most probably) creates it)
 */
Element2D* Element2D::createNullElement()
{
  if (likely(Element2D::nullElement == NULL))
  {
    Element2D::nullElement = new Element2D(NULL, E2D_LAYER_BELOW_ALL, E2D_PARENT_MODE_DEFAULT | E2D_REPARENT_TO_NULL);
    Element2D::nullElement->setName("NullElement");
  }
  return Element2D::nullElement;
}


/**
 * !! PRIVATE FUNCTION
 * @brief checks the upward integrity (e.g if Element2D is somewhere up the Node tree.)
 * @param checkParent the Parent to check.
 * @returns true if the integrity-check succeeds, false otherwise.
 *
 * If there is a second occurence of checkParent before NULL, then a loop could get
 * into the Tree, and we do not want this.
 */
bool Element2D::checkIntegrity(const Element2D* checkParent) const
{
  const Element2D* parent = this;
  while ( (parent = parent->getParent2D()) != NULL)
    if (unlikely(parent == checkParent))
      return false;
  return true;
}


/**
 * @brief updates the absCoordinate/absDirection
 * @param dt The time passed since the last update

   this is used to go through the parent-tree to update all the absolute coordinates
   and directions. this update should be done by the engine, so you don't have to
   worry, normaly...
 */
void Element2D::update2D (float dt)
{
  // setting the Position of this 2D-Element.
  if( likely(this->parent != NULL))
  {
    // movement for nodes with smoothMove enabled
    if (unlikely(this->toCoordinate != NULL))
    {
      Vector2D moveVect = (*this->toCoordinate - this->relCoordinate) *fabsf(dt)*bias;

      if (likely(moveVect.len() >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftCoor2D(moveVect);
      }
      else
      {
        Vector2D tmp = *this->toCoordinate;
        this->setRelCoor2D(tmp);
        PRINTF(5)("SmoothMove of %s finished\n", this->getName());
      }
    }
    if (unlikely(this->toDirection != NULL))
    {
      float rotFlot = (*this->toDirection - this->relDirection) *fabsf(dt)*bias;
      if (likely(fabsf(rotFlot) >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftDir2D(rotFlot);
      }
      else
      {
        float tmp = *this->toDirection;
        this->setRelDir2D(tmp);
        PRINTF(5)("SmoothRotate of %s finished\n", this->getName());
      }
    }
    if (unlikely(this->toSize != NULL))
    {
      Vector2D shiftSize = (*this->toSize - Vector2D(this->sizeX, this->sizeY)) *fabsf(dt)*bias;
      if (likely((shiftSize).len() >= .001))//PNODE_ITERATION_DELTA))
      {
        this->sizeX += shiftSize.x;
        this->sizeY += shiftSize.y;
      }
      else
      {
        delete this->toSize;
        this->toSize = NULL;
        PRINTF(5)("SmoothRotate of %s finished\n", this->getName());
      }
    }

    // MAIN UPDATE /////////////////////////////////////
    this->lastAbsCoordinate = this->absCoordinate;

    PRINTF(5)("Element2D::update - %s - (%f, %f)\n", this->getName(), this->absCoordinate.x, this->absCoordinate.y);


    if( this->parentMode & E2D_PARENT_LOCAL_ROTATE && this->bRelDirChanged)
    {
      /* update the current absDirection - remember * means rotation around sth.*/
      this->prevRelDirection = this->relDirection;
      this->absDirection = this->relDirection + parent->getAbsDir2D();;
    }


    if (unlikely(this->alignment & E2D_ALIGN_SCREEN_CENTER && this->bRelCoorChanged))
    {
      this->prevRelCoordinate = this->relCoordinate;
      this->absCoordinate.x = .5 + this->relCoordinate.x;
      this->absCoordinate.y = .5 + this->relCoordinate.y;
    }
    else if (unlikely(this->bindNode != NULL))
    {
      if (State::getCamera()->distance(this->bindNode) < 0)
        this->bCurrentlyVisible = false;
      else
      {
        this->bCurrentlyVisible = true;
      }

      GLdouble projectPos[3] = {0.0, 0.0, 0.0};
      gluProject(this->bindNode->getAbsCoor().x,
                 this->bindNode->getAbsCoor().y,
                 this->bindNode->getAbsCoor().z,
                 GraphicsEngine::modMat,
                 GraphicsEngine::projMat,
                 GraphicsEngine::viewPort,
                 projectPos,
                 projectPos+1,
                 projectPos+2);
      //       printf("%s::%s  == %f %f %f :: %f %f\n", this->getClassName(), this->getName(),
      //              this->bindNode->getAbsCoor().x,
      //              this->bindNode->getAbsCoor().y,
      //              this->bindNode->getAbsCoor().z,
      //              projectPos[0],
      //              projectPos[1]);

      this->prevRelCoordinate.x = this->absCoordinate.x = projectPos[0] /* /(float)GraphicsEngine::getInstance()->getResolutionX() */ + this->relCoordinate.x;
      this->prevRelCoordinate.y = this->absCoordinate.y = (float)GraphicsEngine::getInstance()->getResolutionY() -  projectPos[1] + this->relCoordinate.y;
      this->bRelCoorChanged = true;
    }
    else
    {
      if(likely(this->parentMode & PNODE_MOVEMENT && this->bRelCoorChanged))
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        this->absCoordinate = this->parent->getAbsCoor2D() + this->relCoordinate;
      }
      else if( this->parentMode & PNODE_ROTATE_MOVEMENT && this->bRelCoorChanged)
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        float sine = sin(this->parent->getAbsDir2D());
        float cose = cos(this->parent->getAbsDir2D());
        //        this->absCoordinate.x = this->relCoordinate.x*cose - this->relCoordinate.y*sine + this->parent->getRelCoor2D().x*(1-cose) +this->parent->getRelCoor2D().y*sine;
        //        this->absCoordinate.y = this->relCoordinate.x*sine + this->relCoordinate.y*cose + this->parent->getRelCoor2D().y*(1-cose) +this->parent->getRelCoor2D().x*sine;

        this->absCoordinate.x = this->parent->getAbsCoor2D().x + (this->relCoordinate.x*cos(this->parent->getAbsDir2D()) - this->relCoordinate.y * sin(this->parent->getAbsDir2D()));
        this->absCoordinate.y = this->parent->getAbsCoor2D().y + (this->relCoordinate.x*sin(this->parent->getAbsDir2D()) + this->relCoordinate.y * cos(this->parent->getAbsDir2D()));

      }
    }
    /////////////////////////////////////////////////
  }
  else
  {
    PRINTF(5)("Element2D::update - (%f, %f)\n", this->absCoordinate.x, this->absCoordinate.y);
    if (this->bRelCoorChanged)
    {
      this->prevRelCoordinate = this->relCoordinate;
      this->absCoordinate = this->relCoordinate;
    }
    if (this->bRelDirChanged)
    {
      this->prevRelDirection = this->relDirection;
      this->absDirection = this->getAbsDir2D() + this->relDirection;
    }
  }


  // UPDATE CHILDREN
  if(!this->children.empty() || this->parentMode & E2D_UPDATE_CHILDREN_IF_INACTIVE)
  {
    std::list<Element2D*>::iterator child;
    for (child = this->children.begin(); child != this->children.end(); child++)
    {
      /* if this node has changed, make sure, that all children are updated also */
      if( likely(this->bRelCoorChanged))
        (*child)->parentCoorChanged2D();
      if( likely(this->bRelDirChanged))
        (*child)->parentDirChanged2D();

      (*child)->update2D(dt);
    }
  }

  // FINISHING PROCESS
  this->velocity = (this->absCoordinate - this->lastAbsCoordinate) / dt;
  this->bRelCoorChanged = false;
  this->bRelDirChanged = false;
}


/**
 * @brief displays some information about this pNode
 * @param depth The deph into which to debug the children of this Element2D to.
 * (0: all children will be debugged, 1: only this Element2D, 2: this and direct children...)
 * @param level The n-th level of the Node we draw (this is internal and only for nice output)
 */
void Element2D::debug2D (unsigned int depth, unsigned int level) const
{
  for (unsigned int i = 0; i < level; i++)
    PRINT(0)(" |");
  if (this->children.size() > 0)
    PRINT(0)(" +");
  else
    PRINT(0)(" -");
  PRINT(0)("E2D(%s::%s);AC:(%0.2f, %0.2f);RC:(%0.2f, %0.2f);AD(%0.2f)->%s;Layer:(%s)\n",
           this->getClassName(),
           this->getName(),
           this->absCoordinate.x,
           this->absCoordinate.y,
           this->relCoordinate.x,
           this->relCoordinate.y,
           this->getAbsDir2D(),
           Element2D::parentingModeToString2D(parentMode),
           Element2D::layer2DToChar(this->layer));

  if (depth >= 2 || depth == 0)
  {
    std::list<Element2D*>::const_iterator child;
    for (child = this->children.begin(); child != this->children.end(); child++)
    {
      if (depth == 0)
        (*child)->debug2D(0, level + 1);
      else
        (*child)->debug2D(depth - 1, level +1);
    }
  }
}

/**
 * @brief ticks the 2d-Element
 * @param dt the time elapsed since the last tick
 *
 * the element only gets tickt, if it is active.
 * Be aware, that this walks through the entire Element2D-tree,
 * searching for Elements to be ticked.
 */
void Element2D::tick2D(float dt)
{
  if (this->bActive)
    this->tick(dt);
  if (this->children.size() > 0)
  {
    std::list<Element2D*>::iterator child;
    for (child = this->children.begin(); child != this->children.end(); child++)
      (*child)->tick2D(dt);
  }
}

/**
 * @brief draws all the Elements from this element2D downwards
 * @param layer the maximal Layer to draw. @see E2D_LAYER
 */
void Element2D::draw2D(E2D_LAYER from, E2D_LAYER to) const
{
  if (this->isVisible())
    this->draw();
  if (this->children.size() > 0)
  {
    std::list<Element2D*>::const_iterator child;
    for (child = this->children.begin(); child != this->children.end(); child++)
      if (likely( (*child)->layer >= from && (*child)->layer <= to))
        (*child)->draw2D(from, to);
  }
}

/**
 * @brief displays the Element2D at its position with its rotation as a Plane.
 */
void Element2D::debugDraw2D(unsigned int depth, float size, Vector color, unsigned int level) const
{
  if (level == 0)
  {
    glPushAttrib(GL_ENABLE_BIT);
    glMatrixMode(GL_MODELVIEW);

    glDisable(GL_LIGHTING);
    glDisable(GL_BLEND);
    glDisable(GL_TEXTURE_2D);
  }

  glPushMatrix();
  /* translate */
  /* rotate */
  glColor3f(color.x, color.y, color.z);

  glTranslatef (this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0);
  glRotatef(this->getAbsDir2D(), 0,0,1);
  glBegin(GL_LINE_LOOP);
  glVertex2f(0, 0);
  glVertex2f(0, +this->getSizeY2D());
  glVertex2f(+this->getSizeX2D(), +this->getSizeY2D());
  glVertex2f(+this->getSizeX2D(), 0);
  glEnd();


  glPopMatrix();
  if (depth >= 2 || depth == 0)
  {
    Vector childColor =  Color::HSVtoRGB(Color::RGBtoHSV(color)+Vector(20,0,.0));
    std::list<Element2D*>::const_iterator child;
    for (child = this->children.begin(); child != this->children.end(); child++)
    {
      // drawing the Dependency graph
      if (this != Element2D::getNullElement())
      {
        glBegin(GL_LINES);
        glColor3f(color.x, color.y, color.z);
        glVertex3f(this->getAbsCoor2D ().x,
                   this->getAbsCoor2D ().y,
                   0);
        glColor3f(childColor.x, childColor.y, childColor.z);
        glVertex3f((*child)->getAbsCoor2D ().x,
                   (*child)->getAbsCoor2D ().y,
                   0);
        glEnd();
      }
      if (depth == 0)
        (*child)->debugDraw2D(0, size, childColor, level+1);
      else
        (*child)->debugDraw2D(depth - 1, size, childColor, level +1);
    }
  }
  if (level == 0)
    glPopAttrib();

}


// helper functions //
/**
 * @brief converts a parentingMode into a string that is the name of it
 * @param parentingMode the ParentingMode to convert
 * @return the converted string
 */
const char* Element2D::parentingModeToString2D(int parentingMode)
{
  if (parentingMode == E2D_PARENT_LOCAL_ROTATE)
    return "local-rotate";
  else if (parentingMode == E2D_PARENT_ROTATE_MOVEMENT)
    return "rotate-movement";
  else if (parentingMode == E2D_PARENT_MOVEMENT)
    return "movement";
  else if (parentingMode == E2D_PARENT_ALL)
    return "all";
  else if (parentingMode == E2D_PARENT_ROTATE_AND_MOVE)
    return "rotate-and-move";
}

/**
 * @brief converts a parenting-mode-string into a int
 * @param parentingMode the string naming the parentingMode
 * @return the int corresponding to the named parentingMode
 */
E2D_PARENT_MODE Element2D::stringToParentingMode2D(const std::string& parentingMode)
{
  if (parentingMode == "local-rotate")
    return (E2D_PARENT_LOCAL_ROTATE);
  else  if (parentingMode == "rotate-movement")
    return (E2D_PARENT_ROTATE_MOVEMENT);
  else  if (parentingMode == "movement")
    return (E2D_PARENT_MOVEMENT);
  else  if (parentingMode == "all")
    return (E2D_PARENT_ALL);
  else  if (parentingMode == "rotate-and-move")
    return (E2D_PARENT_ROTATE_AND_MOVE);
}

/**
 * @brief converts a layer into its corresponding string
 * @param layer the layer to get the name-String of.
 * @returns the Name of the Layer (on error the default-layer-string is returned)
 */
const char* Element2D::layer2DToChar(E2D_LAYER layer)
{
  switch(layer)
  {
    case E2D_LAYER_ABOVE_ALL:
      return "above-all";
    case E2D_LAYER_TOP:
      return "top";
    case E2D_LAYER_MEDIUM:
      return "medium";
    case E2D_LAYER_BOTTOM:
      return "bottom";
    case E2D_LAYER_BELOW_ALL:
      return "below-all";
    default:
      assert (false);
      return layer2DToChar(E2D_DEFAULT_LAYER);
  }
}

/**
 * @brief converts a String holding a actual Layer
 * @param layer the String to convert into a Layer2D
 * @returns the E2D_LAYER on success, E2D_DEFAULT_LAYER on error.
 */
E2D_LAYER Element2D::charToLayer2D(const std::string& layer)
{
  if (layer == "above-all")
    return (E2D_LAYER_ABOVE_ALL);
  if (layer == "top")
    return (E2D_LAYER_TOP);
  else  if (layer == "medium")
    return (E2D_LAYER_MEDIUM);
  else  if (layer == "bottom")
    return (E2D_LAYER_BOTTOM);
  else  if (layer == "below-all")
    return (E2D_LAYER_BELOW_ALL);
  else
    return (E2D_DEFAULT_LAYER);
}
