/*
	orxonox - the future of 3D-vertical-scrollers

	Copyright (C) 2004 orx

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

### File Specific:
	main-programmer: hdavid, amaechler
*/

#include "cloud_effect.h"

#include "util/loading/load_param.h"
#include "util/loading/factory.h"

#include "glincl.h"
//#include "graphics_engine.h"
#include <math.h>

#include "parser/tinyxml/tinyxml.h"

using namespace std;

CREATE_FACTORY(CloudEffect, CL_CLOUD_EFFECT);

CloudEffect::CloudEffect(const TiXmlElement* root)
{
	this->setClassID(CL_CLOUD_EFFECT, "CloudEffect");

	this->init();

	if (root != NULL)
		this->loadParams(root);

	this->activate();
}

CloudEffect::~CloudEffect()
{
	this->deactivate();
}

void CloudEffect::loadParams(const TiXmlElement* root)
{
	WeatherEffect::loadParams(root);

	LoadParam(root, "animSpeed", this, CloudEffect, setCloudAnimation);

}


bool CloudEffect::init()
{
	// default values
	this->cloudAnimTimeStep = 0;

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	// Generate noise map a
	CloudEffect::genNoiseMap(cloudMap32_a);
	
	if (this->cloudAnimTimeStep > 0) {
		// Generate noise map b
		CloudEffect::genNoiseMap(cloudMap32_b);
	}
}

bool CloudEffect::activate()
{
	PRINTF(0)( "Activating CloudEffect\n");
	if (this->cloudAnimTimeStep == 0) {
		for (int i = 0; i < 32*32; i++)
			cloudMap32_c[i] = cloudMap32_a[i];

		CloudEffect::overlapOctaves();
		CloudEffect::expFilter();
		CloudEffect::genCloudTexture();
	}
}

bool CloudEffect::deactivate()
{
	PRINTF(0)("Deactivating CloudEffect\n");
}

void CloudEffect::genCloudTexture() {
	for(int i=0; i<256; i++)
		for(int j=0; j<256; j++) 
		{
			float color = cloudMap256[i*256+j]; 
			cloudTexture[i][j][0] = (char) color;
			cloudTexture[i][j][1] = (char) color;
			cloudTexture[i][j][2] = (char) color;
		}

	glGenTextures(2, &texID[0]);
	glBindTexture(GL_TEXTURE_2D, texID[0]);
	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, 256, 256, GL_RGB, GL_UNSIGNED_BYTE, cloudTexture);
}

void CloudEffect::calcAnimMap(float timer) {

	for (int x=0; x<32*32; x++)
		cloudMap32_c[x] = (cloudMap32_a[x] * ((this->cloudAnimTimeStep - timer) / this->cloudAnimTimeStep)) + (cloudMap32_b[x] * (timer / this->cloudAnimTimeStep));

}

void CloudEffect::draw() const
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glPushMatrix();
	glEnable(GL_TEXTURE_2D);

	glBindTexture(GL_TEXTURE_2D, texID[0]);

	// FIXME : Bind this to the sky - how do I do this?
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(20, 20,  60);	// Bottom Left Of The Texture and Quad
		glTexCoord2f(1.0f, 0.0f); glVertex3f(60, 20,  60);	// Bottom Right Of The Texture and Quad
		glTexCoord2f(1.0f, 1.0f); glVertex3f(60, 60,  60);	// Top Right Of The Texture and Quad
		glTexCoord2f(0.0f, 1.0f); glVertex3f(20, 60,  60);	// Top Left Of The Texture and Quad
	glEnd(); 

	glPopMatrix();
}

void CloudEffect::tick (float dt)
{
	if (this->cloudAnimTimeStep > 0) {
		if (timer >= this->cloudAnimTimeStep) {
			timer -= this->cloudAnimTimeStep;
			for (int i = 0; i < 32*32; i++)
				cloudMap32_a[i] = cloudMap32_b[i];
			CloudEffect::genNoiseMap(cloudMap32_b);
		}

		//map32anim = (map32a * (10 - timer)) + (map32b * timer);
		CloudEffect::calcAnimMap(timer);

		CloudEffect::overlapOctaves();
		CloudEffect::expFilter();
		CloudEffect::genCloudTexture();

		timer += dt;
	}
}

/* 
	Random noise generator
*/
float CloudEffect::noise(int x, int y, int random)
{
		int n = x + y * 57 + random * 131;
		n = (n<<13) ^ n;
		return (1.0f - ( (n * (n * n * 15731 + 789221) +
						1376312589)&0x7fffffff)* 0.000000000931322574615478515625f);
}

/*
	Set noise for the 32*32 noise map:
*/
void CloudEffect::genNoiseMap(float  *map)
{
	float temp[34][34];

	int random = rand() % 5000;

	for (int y = 1; y < 33; y++)
		for (int x = 1; x < 33; x++)
			temp[x][y] = 128.0f + CloudEffect::noise(x,  y,  random) * 128.0f;

	// Seamless cloud
	for (int x=1; x<33; x++)
	{
		temp[0][x] = temp[32][x];
		temp[33][x] = temp[1][x];
		temp[x][0] = temp[x][32];
		temp[x][33] = temp[x][1];
	}
	temp[0][0] = temp[32][32];
	temp[33][33] = temp[1][1];
	temp[0][33] = temp[32][1];
	temp[33][0] = temp[1][32];

	// We mirror the side and corner elements so our final cloud will be seamless without any ugly borders showing.
	for (int y=1; y<33; y++)
		for (int x=1; x<33; x++)
		{
			float center = temp[x][y]/4.0f;
			float sides = (temp[x+1][y] + temp[x-1][y] + temp[x][y+1] + temp[x][y-1])/8.0f;
			float corners = (temp[x+1][y+1] + temp[x+1][y-1] + temp[x-1][y+1] + temp[x-1][y-1])/16.0f;

			map[((x-1)*32) + (y-1)] = center + sides + corners;
		}
}

/*
	Interpolation - average the value of each pixel value with that of its neighbors' values.
*/
float CloudEffect::interpolate(float x, float y, float  *map)
{
	int Xint = (int)x;
	int Yint = (int)y;

	float Xfrac = x - Xint;
	float Yfrac = y - Yint;

	int X0 = Xint % 32;
	int Y0 = Yint % 32;
	int X1 = (Xint + 1) % 32;
	int Y1 = (Yint + 1) % 32;

	float bot = map[X0*32 + Y0] + Xfrac * (map[X1*32 + Y0] - map[X0*32 + Y0]);
	float top = map[X0*32 + Y1] + Xfrac * (map[X1*32 +  Y1] - map[X0*32 + Y1]);

	return (bot + Yfrac * (top - bot));
}


/*
	Octaves are overlapped together to give cloud more turbulence. We will use four octaves for our cloud.
*/
void CloudEffect::overlapOctaves()
{
	for (int x=0; x<256*256; x++)
	{
		cloudMap256[x] = 0;
	}

	for (int octave=0; octave<4; octave++)
		for (int x=0; x<256; x++)
			for (int y=0; y<256; y++)
			{
				float scale = 1 / pow(2.0f, (float) 3-octave);
				float noise = CloudEffect::interpolate(x*scale, y*scale , cloudMap32_c);

				//The octaves are added together with the proper weight factors.
				//You could replace pow(2, i) with 1<<i for faster computation
				cloudMap256[(y*256) + x] += noise / pow(2.0f, (float) octave);
			}
}


/*
	Filter the noise with exponential function
*/
void CloudEffect::expFilter()
{
	float cover = 20.0f;
	float sharpness = 0.95f;

	for (int x=0; x<256*256; x++)
	{
		float c = cloudMap256[x] - (255.0f-cover);
		if (c<0)     c = 0;
		cloudMap256[x] = 255.0f - ((float)(pow(sharpness, c))*255.0f);
	}
}


