/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "texture.h"

#include "debug.h"
#include "compiler.h"

// INCLUDING SDL_Image
#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif


TextureData::TextureData()
{
  this->bAlpha = false;
  this->texture = 0;
  this->image = NULL;
}


/**
 * @brief Destructor of a Texture
 *
 *  Frees Data, and deletes the textures from GL
 */
TextureData::~TextureData()
{
  if (this->texture != 0)
    glDeleteTextures(1, &this->texture);
  if (this->image != NULL)
    SDL_FreeSurface(this->image);
}


/**
 * @brief Loads an SDL_Surface.
 */
bool TextureData::loadSurface(SDL_Surface* surface, GLenum target)
{
  if (Texture::getTextureEnableState())
  {
    SDL_Surface* newSurf = Texture::prepareSurface(surface, this->bAlpha);
    if (newSurf != NULL)
    {
      this->setSurface(newSurf);
      this->setTexture(Texture::loadTexToGL(newSurf, target));
      return true;
    }
  }
  return false;
}



/**
 * @brief set the surface this Texture handles
 * @param newSurface the new Surface to set as the image for this Texture.
 *
 * This deletes the old version of the stored Texture,
 * and sets the newly given Surface as current.
 */
bool TextureData::setSurface(SDL_Surface* newSurface)
{
  if (this->image != NULL)
    SDL_FreeSurface(this->image);

  this->image = newSurface;

  return (this->image != NULL);
}



bool TextureData::setTexture(GLuint texture)
{
     // unload the old Texture.
  if (this->texture != 0 && glIsTexture(this->getTexture()))
  {
    glDeleteTextures(1, &this->texture);
  }
  this->texture = texture;
  return (texture != 0);
}

Texture::Texture()
{
  this->init();
}


Texture::Texture(const Texture& texture)
  : data(texture.data)
{
  this->setClassID(CL_TEXTURE, "Texture");
  this->priority = 0.5;
}


Texture::Texture(GLenum target)
{
  this->init();
  GLuint texture;
  Texture::generateTexture(texture, target);
  this->data->setTexture(texture);
}

/**
 *  Constructor for a Texture
 */
Texture::Texture(const std::string& imageName, GLenum target)
{
  this->init();

  if (!imageName.empty())
  {
    this->setName(imageName);
    this->loadImage(imageName, target);
  }
}



Texture::Texture(SDL_Surface* surface, GLenum target)
{
  this->init();

  if(surface != NULL)
  {
    this->data->loadSurface(surface, target);
  }
}

void Texture::init()
{
  this->setClassID(CL_TEXTURE, "Texture");

  this->data = CountPointer<TextureData>(new TextureData());

  this->priority = 0.5;
}

/**
 * @brief Destructor of a Texture
 *
 * Frees Data, and deletes the textures from GL
 */
Texture::~Texture()
{
}


/**
 * @brief loads an Image from a file to a Texture
 * @param imageName The image to load
*/
bool Texture::loadImage(const std::string& imageName, GLenum target)
{
  if (Texture::texturesEnabled)
  {
    if (!imageName.empty())
    {
      SDL_Surface* tmpSurf;

      // load the new Image to memory
      tmpSurf = IMG_Load(imageName.c_str());
      if(tmpSurf != NULL)
      {
        this->data->loadSurface(tmpSurf, target);
        SDL_FreeSurface(tmpSurf);
        return true;
      }
      else
      {
        PRINTF(1)("IMG_Load: %s\n", IMG_GetError());
        this->setTexture(0);
        return false;
      }
    }
    else
    {
      PRINTF(2)("Image-Name not specified\n");
      return false;
    }
  }
  return false;
}

/**
 * @brief rebuilds the texture.
 *
 * reloads the Texture from Memory to OpenGL.
 */
bool Texture::rebuild()
{
  this->data->setTexture(0);

  if (this->data->getStoredImage() != NULL)
  {
    PRINTF(3)("Reloading Texture of %s '%s'\n", this->getClassName(), this->getName());
    this->setTexture(Texture::loadTexToGL(this->data->getStoredImage()));
  }
}

bool Texture::texturesEnabled = true;

/**
 * @brief enables, disables textures
 * @param texturesEnabled true if the textures should be enabled
 */
void Texture::setTextureEnableState(bool texturesEnabled)
{
  Texture::texturesEnabled = texturesEnabled;
}


//////////////////////////////////////
// UTILITY FUNCTIONALITY OF TEXTURE //
//////////////////////////////////////
/**
 * @brief converts surface to a new SDL_Surface, that is loadable by openGL
 * @param surface the Surface to convert
 * @param hasAlpha if the newly created Surface has an alpha channel, true is returned otherwise false.
 * @returns a !!new!! Surface, that is loadable by openGL.
 */
SDL_Surface* Texture::prepareSurface(SDL_Surface* surface, bool& hasAlpha)
{
  assert(surface != NULL);
  PRINTF(4)("Loading texture to OpenGL-Environment.\n");

  SDL_Surface* retSurface;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;

  hasAlpha = false;
  int pixelDepth = 24;

  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA | SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA )
  {
    SDL_SetAlpha(surface, 0, 0);
    hasAlpha = true;
    pixelDepth = 32;
  }

  retSurface = SDL_CreateRGBSurface(SDL_HWSURFACE,
                                    surface->w, surface->h,
                                    pixelDepth,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                    0x000000FF,
                                    0x0000FF00,
                                    0x00FF0000,
                                    0xFF000000
#else
                                    0xFF000000,
                                    0x00FF0000,
                                    0x0000FF00,
                                    0x000000FF
#endif
                                   );
  if ( retSurface == NULL )
    return NULL;

  /* Copy the surface into the GL texture this->data->getStoredImage() */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, retSurface, &area);

  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA )
  {
    SDL_SetAlpha(surface, saved_flags | SDL_OPENGL, saved_alpha);
    hasAlpha = true;
  }

  return (retSurface);
}


/**
 * @brief Loads a Texture to the openGL-environment.
 * @param surface the Image to load to openGL
 * @returns The ID of the texture.
 */
GLuint Texture::loadTexToGL (const SDL_Surface* surface, GLenum target)
{
  //   if (this->data->getTexture() != 0 && glIsTexture(this->data->getTexture()))
  //     glDeleteTextures(1, &this->data->getTexture());
  //   this->data->getTexture() = 0;
  assert(surface != NULL);

  int      errorCode = 0;           //!< the error code for the texture loading functions
  GLuint   texture = 0;             //!< the OpenGL texture handle
  int      mipmapLevel = 0;         //!< the maximum mipmap level for this texture
  int      mipmapWidth = 0;         //!< the width of the mipmap
  int      mipmapHight = 0;         //!< the height of the mipmap
  GLenum   format = GL_RGB;
  if (surface->format->BitsPerPixel == 32)
  {
    format = GL_RGBA;
    assert(surface->format->BitsPerPixel == 32);
  }
  else
  {
    assert(surface->format->BitsPerPixel == 24);
  }

  /* Create an OpenGL texture for the this->data->getStoredImage() */
  Texture::generateTexture(texture, target);

//   glTexImage2D(target,  0,  format,
//                surface->w,  surface->h,
//                0, format,  GL_UNSIGNED_BYTE,
//                surface->pixels);

///  glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_REPEAT);
///  glTexParameteri(target, GL_TEXTURE_WRAP_R, GL_REPEAT);

  /// TODO CHECK THIS BACK in
  //glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_PRIORITY, this->priority);

  /* build the Texture  OpenGL V >= 1.1 */

  //  printf("%s, w:%d h:%d, 0x%x\n", this->getName(), surface->w, surface->h, target);

  // build the MipMaps automaticaly
  errorCode = gluBuild2DMipmaps(target, format,
                                surface->w,  surface->h,
                                format,  GL_UNSIGNED_BYTE,
                                surface->pixels
                               );
  if(unlikely(errorCode != 0))
    PRINTF(1)("Error while loading texture (mipmap generation), gluBuild2DMipmaps returned %i\n", errorCode);

  return texture;
}

void Texture::generateTexture(GLuint& texture, GLenum target)
{
  if (texture == 0 && !glIsTexture(texture))
  {
    glGenTextures(1, &texture);
  }
  glBindTexture(target, texture);

  glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(target, GL_TEXTURE_WRAP_R, GL_REPEAT);

  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  /* control the mipmap levels */
  glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_MIN_LOD, 5);
  glTexParameterf(GL_TEXTURE_ENV, GL_TEXTURE_MAX_LOD, 0);
}
