/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christoph Renner
   co-programmer: ...
*/

#include "connection_monitor.h"
#include "network_log.h"

#include <debug.h>
#include <SDL/SDL.h>
#include <string.h>

/* using namespace std is default, this needs to be here */
using namespace std;

ConnectionMonitor::ConnectionMonitor( int userId )
{
  /* set the class id for the base object and add ist to class list*/
  this->setClassID(CL_CONNECTION_MONITOR, "ConnectionMonitor");
  
  this->userId = userId;
  this->ping = 0;
  this->incomingUnzippedBandWidth = 0;
  this->outgoingUnzippedBandWidth = 0;
  this->nIncomingPackets = 0;
  this->nOutgoingPackets = 0;
  
  this->lastPacketTick = 0;
}

ConnectionMonitor::~ConnectionMonitor( )
{
}

void ConnectionMonitor::processUnzippedOutgoingPacket( byte * data, int length, int stateId )
{
  int tick = SDL_GetTicks();
  
  nOutgoingPackets++;
  
  // for ping calculation
  sentStateTicks[stateId] = tick;
  
  // calculate bandwidth
  outgoingUnzippedPacketHistory[tick] = length;
  outgoingUnzippedBandWidth = calculateBandWidth( outgoingUnzippedPacketHistory, tick );
  
  NETPRINTF(n)("UPSTREAM: user: %d bandwidth %f\n", userId, outgoingUnzippedBandWidth );
  
  // count zero bytes
  int nZeroBytes = 0;
  
  for ( int i = 0; i < length; i++ )
    if ( data[i] == '\0' )
      nZeroBytes++;
  
  NETPRINTF(n)( "ZEROBYTES: %d (%f%%)\n", nZeroBytes, ((float)100)*nZeroBytes/length );
}

void ConnectionMonitor::processUnzippedIncomingPacket( byte * data, int length, int stateId, int ackedState )
{
  int tick = SDL_GetTicks();
  
  nIncomingPackets++;
  
  lastPacketTick = tick;
  
  // calculate ping
  if ( sentStateTicks.find( ackedState ) != sentStateTicks.end() )
  {
    ackDelay.push_back( tick - sentStateTicks[ackedState] );
  }
      
  while ( sentStateTicks.begin()->first <= ackedState )
    sentStateTicks.erase( sentStateTicks.begin() );
      
  while ( ackDelay.size() > N_PACKETS_FOR_PING )
    ackDelay.erase( ackDelay.begin() );
      
  ping = 0;
      
  for ( std::list<int>::iterator it = ackDelay.begin(); it != ackDelay.end(); it++ )
    ping += *it;
      
  if ( ackDelay.size() == 0 )
    ping = -1;
  else
    ping /= ackDelay.size();
      
  NETPRINTF(n)("PING: user: %d ping: %d\n", userId, ping );
  
  // calculate bandwidth
  incomingUnzippedPacketHistory[tick] = length;
  incomingUnzippedBandWidth = calculateBandWidth( incomingUnzippedPacketHistory, tick );
  
  NETPRINTF(n)("DOWNSTREAM: user: %d bandwidth %f\n", userId, incomingUnzippedBandWidth );
  
}

float ConnectionMonitor::calculateBandWidth( std::map< int, int > packetHistory, int tick )
{
  // delete old packets
  while ( packetHistory.begin()->first < tick - MSECS_TO_CALC_BWIDTH )
    packetHistory.erase( packetHistory.begin() );
  
  float res = 0.0f;
  
  for ( std::map<int,int>::iterator it = packetHistory.begin(); it != packetHistory.end(); it++ )
  {
    res += it->second;
  }
  
  if ( packetHistory.size() <= 1 || tick - packetHistory.begin()->first == 0 )
    res = 0;
  else
    res /= (float)((tick - packetHistory.begin()->first)*( 1 + 1/((float)(packetHistory.size()-1)) ));
  
  res *= 1000;
  
  return res;
}


