/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "multi_line_text.h"
#include "font.h"

using namespace std;

/**
 * @brief creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
MultiLineText::MultiLineText(const std::string& fontFile, unsigned int textSize, float lineWidth)
  : Text(fontFile, textSize)
{
  this->setClassID(CL_MULTI_LINE_TEXT, "MultiLineText");

  this->lineSpacing = 1.0;
  this->lineCount = 0;
  this->setLineWidth(lineWidth);
}

/**
 * @brief sets the maximum Line width
 * @param lineWidth the maximum lineWidth.
 */
void MultiLineText::setLineWidth(float lineWidth)
{
  this->lineWidth = lineWidth;
  this->setSizeX2D(lineWidth);
  this->setupTextWidth();
}


/**
 * @param lineSpacing: the Spacing between the lines
 */
void MultiLineText::setLineSpacing(float lineSpacing)
{
   this->lineSpacing = lineSpacing;
   this->setupTextWidth();
}


/**
 * @brief draws the Text
 */
void MultiLineText::draw() const
{
  if (unlikely(this->getText().empty()))
    return;
  glPushMatrix();
  // transform for alignment.
  // TODO make the Stuff with the alignment
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->getSizeX2D(), 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->getSizeX2D()/2, 0, 0);

  // drawing this Text.
  // setting the Blending effects
  glColor4f(this->getColor().x, this->getColor().y, this->getColor().z, this->getBlending());
  glEnable(GL_BLEND);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, GL_MODULATE );

  glBindTexture(GL_TEXTURE_2D, this->getFont()->getTexture());
  glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0);
  glRotatef(this->getAbsDir2D(), 0, 0, 1);

  Glyph* tmpGlyph;
  float posX = 0.0f;
  float posY = 0.0f;
  unsigned int lineNumber = 0;

  glBegin(GL_QUADS);
  for (unsigned int i = 0; i < this->getText().size(); ++i)
  {
    if (unlikely(this->lineEnds.size() > lineNumber && i == this->lineEnds[lineNumber]))
    {
      // go to the next Line.
      ++lineNumber;
      posX = 10.0f;
      posY += this->lineSpacing + this->getSize(); //this->getFont()->getMaxHeight();
    }

    if(likely((tmpGlyph = this->getFont()->getGlyphArray()[this->getText()[i]]) != NULL))
    {
      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->maxX*this->getSize(), posY);

      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->maxX*this->getSize(), posY + this->getSize());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->minX*this->getSize(), posY+ this->getSize());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->minX*this->getSize(), posY);

      posX += tmpGlyph->advance * this->getSize();
    }
  }
  glEnd();
  glPopMatrix();
}


/**
 * @brief setting up the Text-Width if DYNAMIC
 */
void MultiLineText::setupTextWidth()
{
  this->lineEnds.clear();
  float width = 0.0f;
  float maxWidth = this->lineWidth / this->getSize();

  for (unsigned int i = 0; i < this->getText().size(); i++)
  {
    if (width > maxWidth || this->getText()[i] == '\n')
    {
      if (likely(i > 0))
      {
        this->lineEnds.push_back( i -1 );
        width = this->getFont()->getGlyphArray()[this->getText()[i-1]]->advance;
      }
      else
        width = 0.0f;
    }

    // Advance the Text.
    if(this->getFont()->getGlyphArray()[this->getText()[i]] != NULL)
      width += this->getFont()->getGlyphArray()[this->getText()[i]]->advance;
  }
  this->lineCount = lineEnds.size() + 1;
  this->setSizeY2D((this->lineEnds.size()+1) * (this->lineSpacing + this->getFont()->getMaxHeight()));
}

void MultiLineText::debug() const
{
 printf("Debug %s::%s: %d lines\n", this->getClassName(), this->getName(), this->getLineCount());

 std::string tmpText = this->getText();
 std::vector<unsigned int> ends = this->lineEnds;
 ends.push_back(tmpText.size());

 unsigned int prev = 0;
  for (unsigned int i = 0; i < ends.size(); i++)
  {
    printf("Line %d: %s\n", i, tmpText.substr(prev, ends[i] - prev).c_str());
    prev = ends[i];
  }
}
