/*!
 * @file shell.h
 * Definition of a on-screen-shell
 *
 * @todo Buffer Display in different Colors for different debug mode.
 * @todo choose color of the Font and the background.
 */

#ifndef _SHELL_H
#define _SHELL_H

#include "element_2d.h"
#include "event_listener.h"

#include "shell_input.h"
#include "material.h"

#define         SHELL_DEFAULT_FONT              "fonts/final_frontier.ttf"
#define         SHELL_DEFAULT_TEXT_COLOR        0.0f, 1.0f, 0.0f, 1.0f
#define         SHELL_DEFAULT_BACKGROUND_COLOR  0.0f, 0.0f, 0.0f, 1.0f

// FORWARD DECLARATION
class MultiLineText;
class ShellInput;
class Material;

namespace OrxShell
{
  //! A class that is able to redirect all output to a openGL-Shell, and that one can use to input some commands
  /**
   * the major idea is, that all the Output can be redirected to the Shell,
   * and does not have to be displayed to the opening Shell, this is good,
   * for developers using Windows, where all output is otherwise redirected
   * to stdout.txt
   *
   * Furthermore the Shell should enable us, to input some simple commands
   * Each Class can check itself in to the Shell, and listen for commands.
   *
   * more info: @see ShellCommand
   * @see shell_command.h
   * @see shell_buffer.h
   * @see shell_input.h
   *
   * !! note in order to keep shellbuffer to a minimal (it is included with
   * !! debug.h) Display of it inside the Shell is located here !!
   */
  class Shell : public Element2D, public EventListener
  {

  public:
    Shell();
    virtual ~Shell();

    void activate();
    void deactivate();
    /** @returns true if the Shell is active, false otherwise. */
    inline bool isActive() const { return this->bActive; };

    void setFont(const std::string& fontFile);
    void setTextSize(unsigned int textSize, unsigned int lineSpacing = 1);
    void setTextColor(float r, float g, float b, float a);
    void setBackgroundColor(float r, float g, float b, float a);
    void setBackgroundImage(const std::string& fileName);

    // BUFFERS
    void setBufferDisplaySize(unsigned int bufferDisplaySize);
    void printToDisplayBuffer(const std::string& text);
    void moveDisplayBuffer(int lineCount);

    void flush();

    void clear();

    // EventListener
    virtual void process(const Event &event);
    // Element2D-functions
    virtual void draw() const;

    void debug() const;

    void testShell() const;
  private:
    void repositionText();
    void applyTextSettings(Text* text);
    void applySettings();
    // helpers //
    Vector2D calculateLinePosition(unsigned int lineNumber);

  private:
    // GENERAL
    bool                        bActive;                //!< If the shell is active.
    unsigned int                shellHeight;            //!< The hight of the Shell in Pixels.
    unsigned int                lineSpacing;            //!< The Spacing between lines.
    unsigned int                textSize;               //!< The size of the text.
    float                       textColor[4];           //!< The text's color [r,g,b,a].
    std::string                 fontFile;               //!< The file containing the font.
    Material                    backgroundMaterial;     //!< A material for the background.

    // HANDLING TEXT INPUT
    ShellInput                  shellInput;             //!< The inputLine of the Shell.
    // BUFFER
    unsigned int                bufferDisplaySize;      //!< The Size of the Display-buffer, in lines (not in characters).
    std::list<MultiLineText*>   bufferText;             //!< A list of stored bufferTexts for the display of the buffer.

    std::list<std::string>::const_iterator  bufferIterator;         //!< used to move through and print the Buffer
  };

}

#endif /* _SHELL_H */
